// SCApp.cpp: The main app window and stuff.

#include "Dialogs.h"
#include <string.h>
#include <wx/utils.h>
#include <wx/file.h>

SCFrame* g_window = NULL;
Images* g_images = NULL;
wxString g_initial[NUMBER_OF_HANDS] = { "", "" };


enum event_ids {
  MENU_SERVER = 100,
  MENU_CLIENT,
  MENU_DISCON,
  MENU_SAVELOG,
  ARENA_BOX,
  INPUT_ENTER,
  END_BUTTON,
  SPELL_BUTTON,
};

IMPLEMENT_APP(SCApp)


bool SCApp::OnInit() {
  bool server = false, client = false;

  SetAppName(wxT(PACKAGE));

  for (int i = 1 ; i < argc ; i++) {
    wxString s(argv[i]);
    if (s.Cmp(wxT("--clearprefs")) == 0) {
      wxConfig::Get()->DeleteAll();
      ExitMainLoop();
      return false;
    } else if (s.Cmp(wxT("--server")) == 0) {
      server = true;
    } else if (s.Cmp(wxT("--client")) == 0) {
      client = true;
    } else if (s.Cmp(wxT("--left")) == 0) {
      g_initial[LEFT_HAND] = argv[i + 1];
      i++;
    } else if (s.Cmp(wxT("--right")) == 0) {
      g_initial[RIGHT_HAND] = argv[i + 1];
      i++;
    }
  }

  g_images = new Images();
  SCFrame::Open(wxT(PACKAGE " " VERSION), wxSize(1024, 768));

  test_serialize();
  Event::test();

  g_window->Show(TRUE);
  SetTopWindow(g_window);
  if (wxConfig::Get()->HasEntry(wxT("player_name")) == false) {
    PrefsDialog pr(true);
    pr.CenterOnParent();
    pr.ShowModal();
  }
  srand(wxGetFreeMemory().ToLong() ^ wxGetUTCTime());

  if (server) { g_window->start_server(); }
  if (client) { g_window->start_client(); }
  return true;
}


int SCApp::OnExit() {
  delete g_images;
  delete wxConfig::Set(NULL);
  return EXIT_SUCCESS;
}


void SCFrame::Open(const wxString& title, const wxSize& size) {
  wxASSERT(g_window == NULL);
  g_window = new SCFrame(title, size);
}


SCFrame::SCFrame(const wxString& title, const wxSize& size)
  : wxFrame((wxFrame*) NULL, -1, title, wxDefaultPosition, size,
	    wxDEFAULT_FRAME_STYLE | wxCLIP_CHILDREN) {
  m_gamewin = NULL;
  m_arenawin = NULL;
  m_spellwin = NULL;
  m_end_btn = NULL;

  // Create the background panel, which covers up the default ugly dark
  // grey background on Windows.
  m_panel = new wxPanel(this, -1);

  // Create the status bar
  m_status = wxT("Not connected.");
  CreateStatusBar();
  SetStatusText(m_status);

  // Create the menu bar
  m_file_menu = new wxMenu;
  m_file_menu->Append(wxID_ABOUT);
  m_file_menu->Append(wxID_PREFERENCES);
#ifndef __WXMAC__
  m_file_menu->AppendSeparator();
#endif
  m_file_menu->Append(MENU_SERVER, wxT("Start &Server...\tCtrl-S"));
  m_file_menu->Append(MENU_CLIENT, wxT("&Join Game...\tCtrl-J"));
  m_file_menu->Append(MENU_DISCON, wxT("&Disconnect"));
  m_file_menu->Append(MENU_SAVELOG, wxT("Save &Log..."));
  m_file_menu->AppendSeparator();
  m_file_menu->Append(wxID_EXIT);
  set_connected(false);

  wxMenu* edit_menu = new wxMenu;
  edit_menu->Append(wxID_CUT);
  edit_menu->Append(wxID_COPY);
  edit_menu->Append(wxID_PASTE);
  edit_menu->Append(wxID_SELECTALL);

  wxMenuBar* menus = new wxMenuBar;
  menus->Append(m_file_menu, wxT("&File"));
  menus->Append(edit_menu, wxT("&Edit"));
  SetMenuBar(menus);

  // Create the input line
  m_input = new wxTextCtrl(m_panel, INPUT_ENTER, wxT(""), wxDefaultPosition,
			   wxDefaultSize, wxTE_PROCESS_ENTER);
  m_input->SetMaxLength(512);      // if it's plenty for IRC, it's plenty for us. :-)

  // Create the log window
  m_log = wxT("");
  m_text = new LogWindow(m_panel, m_input, wxDefaultPosition, wxSize(300, 100));

  // Create the gesture window
  m_gamewin = new GestureWindow(m_panel);
  m_gamewin->SetClickable(false);

  // Create the buttons and make them the same size.
  m_spell_btn = new wxButton(m_panel, SPELL_BUTTON, wxT("mmSpell Listmm"), wxDefaultPosition);
  m_end_btn = new wxButton(m_panel, END_BUTTON, wxT("Send Moves"), wxDefaultPosition);
  m_end_btn->Enable(FALSE);
  wxSize foo(find_max_widget_size(m_spell_btn->GetSize(), m_end_btn->GetSize()));
  m_spell_btn->SetMinSize(foo);
  m_end_btn->SetMinSize(foo);
  m_spell_btn->SetLabel("Spell List");

  // Miscellaneous other stuff
  m_arenawin = new ArenaList(m_panel, ARENA_BOX);
  SetIcon(g_images->get_icon());

  // Fit everything in a pretty layout.
  /*
    + wxVERTICAL               = winsizer
      - wxHORIZONTAL           = topsizer
        - wxVERTICAL           = chatsizer
          - wxTextArea         = m_text
          - wxTextArea         = m_input
        - wxVERTICAL           = m_gamesizer
          - GestureWindow      = m_gamewin
          - wxHORIZONTAL       = midsizer
            - wxVERTICAL       = buttonsizer
              - wxButton       = m_end_btn
              - wxButton       = m_spell_btn
            - ArenaList        = m_arenawin
      - wxStaticBoxSizer       = boxsizer
        - wxFlexGridSizer      = questions
          - wxStaticText       = m_qtext
          - wxComboBox         = m_qboxes
  */
  wxBoxSizer* framesizer = new wxBoxSizer(wxVERTICAL);
  wxBoxSizer* winsizer = new wxBoxSizer(wxVERTICAL);
  wxBoxSizer* topsizer = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer* chatsizer = new wxBoxSizer(wxVERTICAL);
  chatsizer->Add(m_text, 1, wxEXPAND | wxALL, 2);
  chatsizer->Add(m_input, 0, wxEXPAND | wxALL, 2);
  topsizer->Add(chatsizer, 0, wxEXPAND | wxALL, 1);

  m_gamesizer = new wxBoxSizer(wxVERTICAL);
  m_gamesizer->Add(m_gamewin, 0, wxEXPAND | wxALL, 4);
  topsizer->Add(m_gamesizer, 1, wxEXPAND | wxALL, 1);

  wxBoxSizer* midsizer = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer* buttonsizer = new wxBoxSizer(wxVERTICAL);
  buttonsizer->Add(m_end_btn, 0, wxALL | wxALIGN_CENTRE, 4);
  buttonsizer->Add(m_spell_btn, 0, wxALL | wxALIGN_CENTRE, 4);
  midsizer->Add(buttonsizer, 0, wxALL, 4);
  midsizer->Add(m_arenawin, 0, wxALL, 4);
  m_gamesizer->Add(midsizer, 0, wxEXPAND | wxALL, 1);

  wxStaticBox* box = new wxStaticBox(m_panel, -1, wxT("Questions"));
  wxStaticBoxSizer* boxsizer = new wxStaticBoxSizer(box, wxHORIZONTAL);
  wxFlexGridSizer* questions = new wxFlexGridSizer(MAX_QUESTIONS, 2, 2, 15);
  questions->AddGrowableCol(0);
  questions->AddGrowableCol(1);

  for (int i = 0 ; i < MAX_QUESTIONS ; i++) {
    m_qtext[i] = new wxStaticText(m_panel, -1, wxT(" "), wxDefaultPosition);
    m_qboxes[i] = new wxComboBox(m_panel, -1, wxT(""), wxDefaultPosition, wxSize(300, 20),
				 0, NULL, wxCB_DROPDOWN | wxCB_READONLY);
    questions->Add(m_qtext[i], 1, wxALL | wxALIGN_LEFT, 0);
    questions->Add(m_qboxes[i], 1, wxALL | wxALIGN_RIGHT, 0);
  }
  boxsizer->Add(questions, 1, wxALL, 0);

  winsizer->Add(topsizer, 1, wxEXPAND | wxALL, 2);
  winsizer->Add(boxsizer, 0, wxEXPAND | wxALL, 2);
  m_panel->SetAutoLayout(TRUE);
  m_panel->SetSizer(winsizer);
  winsizer->Fit(this);
  winsizer->SetSizeHints(this);
  framesizer->Add(m_panel, 1, wxEXPAND | wxALL, 0);
  SetSizer(framesizer);
  Centre();

  // We have to wait until after the size calculations to hide these, or
  // else the layout goes all haywire and the qboxes end up at the origin.
  // (New bug when I upgraded from wxW 2.8.7 to SVN 2008-04-12.)
 for (int i = 0 ; i < MAX_QUESTIONS ; i++) {
    m_qboxes[i]->Hide();
  }
}


SCFrame::~SCFrame() {
  wxASSERT(g_window != NULL);
  if (g_client) delete g_client;
  if (g_server) delete g_server;
  if (m_spellwin) delete m_spellwin;
  delete m_text;
  delete m_input;
  g_window = NULL;
}


void SCFrame::start_server() {
  if (Server::Open(GET_SERVER_NAME(), GET_MAX_PLAYERS(), GET_MY_SERVER_PORT())) {
    wxIPV4address addr;
    addr.LocalHost();
    addr.Service(GET_MY_SERVER_PORT());

    if (Client::Open(GET_CLIENT_NAME(), addr)) {
      g_window->set_connected(true);
      refresh();
    } else {
      delete g_server;
    }
  }
}


void SCFrame::start_client() {
  wxIPV4address addr;
  addr.Hostname(GET_SERVER_ADDR());
  addr.Service(GET_SERVER_PORT());

  if (Client::Open(GET_CLIENT_NAME(), addr)) {
    g_window->set_connected(true);
    refresh();
  }
}


void SCFrame::set_connected(bool is_connected) {
  if (is_connected) {
    wxASSERT(g_client != NULL);
  } else {
    if (g_client) delete g_client;
    if (g_server) delete g_server;
    SetStatus(wxT("Not connected."));
    if (m_end_btn) m_end_btn->Enable(FALSE);
  }

  m_file_menu->FindItem(MENU_SERVER)->Enable(! is_connected);
  m_file_menu->FindItem(MENU_CLIENT)->Enable(! is_connected);
  m_file_menu->FindItem(MENU_DISCON)->Enable(is_connected);
  refresh();
}


void SCFrame::turn_started() {
  wxASSERT(g_client != NULL);
  if (g_client->state().phase() == ANSWERS_PHASE) {
    if (m_qboxes[0]->GetCount() > 0) {
      m_gamewin->SetClickable(false);
      m_end_btn->SetLabel(wxT("Send Answers"));
      m_end_btn->Enable(true);
    } else {
      goto not_active;
    }
  } else if (g_client->state().phase() == GESTURES_PHASE && g_client->state().me()->active()) {
    m_end_btn->SetLabel(wxT("Send Moves"));
    m_end_btn->Enable(true);
    m_gamewin->SetClickable(true);

  } else {
  not_active:
    m_gamewin->SetClickable(false);
    m_end_btn->SetLabel(wxT("Please wait..."));
    m_end_btn->Enable(false);
  }
}


void SCFrame::game_over() {
  m_end_btn->SetLabel(wxT("Game Over"));
  m_end_btn->Enable(false);
  m_gamewin->SetClickable(false);
  clear_questions();
}


void SCFrame::log(const wxChar* fmt, ...) {
  wxString str;
  va_list ap;

  wxASSERT(fmt != NULL);
  va_start(ap, fmt);
  REQUIRE(str.PrintfV(fmt, ap) >= 0);
  va_end(ap);

  log(str);
}


void SCFrame::log(const wxString& str) {
  m_text->AppendText(str);
  m_text->AppendText(wxT("\n"));
  log_hidden(str);
}


void SCFrame::log_hidden(const wxChar* fmt, ...) {
  wxString str;
  va_list ap;

  wxASSERT(fmt != NULL);
  va_start(ap, fmt);
  REQUIRE(str.PrintfV(fmt, ap) >= 0);
  va_end(ap);

  log_hidden(str);
}


void SCFrame::log_hidden(const wxString& str) {
  m_log += str;
  m_log += '\n';
}


void SCFrame::refresh() {
  if (g_server && g_server->state().game_state() == GAME_NOT_STARTED) {
    m_end_btn->SetLabel(wxT("Start Game"));
    m_end_btn->Enable(g_server->state().player_count() > 1);
  }
  if (m_gamewin) {
    m_gamewin->Refresh();
  }
  if (m_arenawin) {
    if (g_client) {
      m_arenawin->UpdateList(g_client->state());
    } else {
      m_arenawin->UpdateList();
    }
  }
  Refresh();
}


void SCFrame::show_question(const Question& q) {
  for (int i = 0 ; i < MAX_QUESTIONS ; i++) {
    if (m_qboxes[i]->GetCount() == 0) {
      m_qtext[i]->SetLabel(q.question());
      for (int x = 0 ; x < q.answer_count() ; x++) {
	m_qboxes[i]->Append(q.answer(x));
      }
      m_qboxes[i]->SetSelection(0);
      m_qboxes[i]->Show();
      return;
    }
  }
  FAIL();  // can't happen -- server sends too many questions at once
}


void SCFrame::clear_questions() {
  for (int i = 0 ; i < MAX_QUESTIONS ; i++) {
    m_qtext[i]->SetLabel(wxT(" "));
    m_qboxes[i]->Clear();
    m_qboxes[i]->Hide();
  }
}


void SCFrame::OnAbout(wxCommandEvent& WXUNUSED(event)) {
  AboutDialog ab;
  ab.CenterOnParent();
  ab.ShowModal();
}


void SCFrame::OnPrefs(wxCommandEvent& WXUNUSED(event)) {
  PrefsDialog pr(false);
  pr.CenterOnParent();
  pr.ShowModal();
}


void SCFrame::OnServer(wxCommandEvent& WXUNUSED(event)) {
  ServerDialog sv;
  sv.CenterOnParent();
  if (sv.ShowModal()) {
    start_server();
  }
}


void SCFrame::OnClient(wxCommandEvent& WXUNUSED(event)) {
  ClientDialog cl;
  cl.CenterOnParent();
  if (cl.ShowModal()) {
    start_client();
  }
}


void SCFrame::OnDiscon(wxCommandEvent& WXUNUSED(event)) {
  set_connected(false);
  SetStatusText(wxT("Not connected."));
}


void SCFrame::OnExit(wxCommandEvent& WXUNUSED(event)) {
  Close(TRUE);
}


void SCFrame::OnCut(wxCommandEvent& WXUNUSED(event)) {
  wxWindow* win = wxWindow::FindFocus();
  if (win->IsKindOf(CLASSINFO(wxTextCtrl))) {
    ((wxTextCtrl*) win)->Cut();
  }
}


void SCFrame::OnCopy(wxCommandEvent& WXUNUSED(event)) {
  wxWindow* win = wxWindow::FindFocus();
  if (win->IsKindOf(CLASSINFO(wxTextCtrl))) {
    ((wxTextCtrl*) win)->Copy();
  }
}


void SCFrame::OnPaste(wxCommandEvent& WXUNUSED(event)) {
  wxWindow* win = wxWindow::FindFocus();
  if (win->IsKindOf(CLASSINFO(wxTextCtrl))) {
    ((wxTextCtrl*) win)->Paste();
  }
}


void SCFrame::OnSelectAll(wxCommandEvent& WXUNUSED(event)) {
  wxWindow* win = wxWindow::FindFocus();
  if (win->IsKindOf(CLASSINFO(wxTextCtrl))) {
    ((wxTextCtrl*) win)->SetSelection(0, ((wxTextCtrl*) win)->GetLastPosition() + 1);
  }
}


void SCFrame::OnSaveLog(wxCommandEvent& WXUNUSED(event)) {
  wxFileDialog fd(this, wxT("Where do you want to save the log?"), GET_LOG_DIR());
  fd.SetWindowStyle(wxFD_SAVE | wxFD_OVERWRITE_PROMPT | wxFD_CHANGE_DIR);
  fd.CenterOnParent();

  if (fd.ShowModal() == wxID_OK && !fd.GetPath().IsEmpty()) {
    wxFile newlog(fd.GetPath(), wxFile::write);

    if (!newlog.IsOpened() || !newlog.Write(m_log)) {
      log(wxT("Couldn't save the log: %s"), wxSysErrorMsg());
    }
    newlog.Close();
    wxConfig::Get()->Write(wxT("log_dir"), wxPathOnly(fd.GetPath()));
  }
}


static wxString _filter_newlines(wxString string) {
  wxUint32 len = string.Length();
  for (wxUint32 i = 0 ; i < len ; i++) {
    if (string[i] == '\n' || string[i] == '\r') {
      StringRemove(string, i, 1);
      len--;  i--;
    }
  }
  return string;
}


void SCFrame::OnChat(wxCommandEvent& event) {
  if (g_client && m_input->GetValue().Length() > 0) {
    wxString str = _filter_newlines(event.GetString());
    g_client->send_event(Event::Chat(str, -1));
  }
  m_input->Clear();
}


void SCFrame::OnSpellList(wxCommandEvent& WXUNUSED(event)) {
  if (m_spellwin == NULL) {
    m_spellwin = new SpellWindow(g_images->get_image(SPELL_LIST));
  }
  m_spellwin->Raise();
}


void SCFrame::OnSpellListDismiss() {
  m_spellwin = NULL;
}


void SCFrame::OnEndButton(wxCommandEvent& WXUNUSED(event)) {
  if (g_server && g_server->state().game_state() == GAME_NOT_STARTED) {
    m_end_btn->SetLabel(wxT("Send Moves"));
    g_server->start_game();

  } else if (g_client) {
    switch (g_client->state().phase()) {
    case GESTURES_PHASE:
      g_client->send_gestures();
      m_end_btn->SetLabel(wxT("Moves Sent"));
      m_end_btn->Enable(false);
      m_gamewin->SetClickable(false);
      break;

    case ANSWERS_PHASE:
      for (int i = 0 ; i < MAX_QUESTIONS ; i++) {
	if (m_qboxes[i]->GetCount() > 0) {
	  wxASSERT(m_qboxes[i]->GetSelection() >= 0);
	  g_client->send_event(Event::Answer(m_qtext[i]->GetLabel(), m_qboxes[i]->GetSelection()));
	}
      }
      clear_questions();
      m_end_btn->SetLabel(wxT("Answers Sent"));
      m_end_btn->Enable(false);
      break;

    default:  FAIL();
    }
  }
}


void SCFrame::OnListBox(wxCommandEvent& event) {
  m_arenawin->SetSelection(event.GetSelection(), FALSE);
}


void SCFrame::SetStatus(const wxString& status) {
  m_status = status;
  SetStatusText(m_status);
}


void SCFrame::ShowStatus(wxMenuEvent& WXUNUSED(event)) {
  SetStatusText(m_status);
}


void SCFrame::OnKeyboard(wxKeyEvent& event) {
  // FIXME: send event to m_input control (ProcessEvent and AddPendingEvent don't work)
  m_input->SetFocus();
}


BEGIN_EVENT_TABLE(SCFrame, wxFrame)
  EVT_MENU(wxID_ABOUT,  SCFrame::OnAbout)
  EVT_MENU(wxID_PREFERENCES,  SCFrame::OnPrefs)
  EVT_MENU(MENU_SERVER, SCFrame::OnServer)
  EVT_MENU(MENU_CLIENT, SCFrame::OnClient)
  EVT_MENU(MENU_DISCON, SCFrame::OnDiscon)
  EVT_MENU(wxID_EXIT,   SCFrame::OnExit)
  EVT_MENU(wxID_CUT,    SCFrame::OnCut)
  EVT_MENU(wxID_COPY,   SCFrame::OnCopy)
  EVT_MENU(wxID_PASTE,  SCFrame::OnPaste)
  EVT_MENU(wxID_SELECTALL, SCFrame::OnSelectAll)
  EVT_MENU(MENU_SAVELOG, SCFrame::OnSaveLog)
  EVT_MENU_HIGHLIGHT_ALL(SCFrame::ShowStatus)
  EVT_TEXT_ENTER(INPUT_ENTER, SCFrame::OnChat)
  EVT_BUTTON(SPELL_BUTTON, SCFrame::OnSpellList)
  EVT_BUTTON(END_BUTTON, SCFrame::OnEndButton)
  EVT_LISTBOX(ARENA_BOX, SCFrame::OnListBox)
  EVT_LISTBOX_DCLICK(ARENA_BOX, SCFrame::OnListBox)
  EVT_KEY_DOWN(SCFrame::OnKeyboard)
  EVT_CHAR(SCFrame::OnKeyboard)
END_EVENT_TABLE()
