// R_Turn.cpp: Rules to handle the turn order and passing of time.

#include "Spellcast.h"
#include "Messages.h"


static int _spell_list_compare(const DoneSpell** a, const DoneSpell** b) {
  if ((*a)->caster < (*b)->caster) {
    return 1;
  } else if ((*a)->caster > (*b)->caster) {
    return -1;
  }
  return 0;
}


void Rules::run_turn_events() {
  static DoneSpellList saved_spells;
  bool alive_ones[ MAX_BEINGS ];

  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL) {
      wxASSERT(player(i)->count_spell_questions() == 0);
    }
  }
  
  // Hasted turns are special. On hasted turns, all completed spells get
  // saved, so that they can go off simultaneously with the completed
  // spells from the following normal turn. Hasted monsters get to
  // attack twice on the next turn. No other turn processing is done
  // this turn.
  if (m_turn_type == TURN_HASTED) {
    for (DoneSpellList::Node* node = m_spells.GetFirst() ; node != NULL ;
	 node = node->GetNext()) {
      DoneSpell* sp = node->GetData();
      wxASSERT(player(sp->caster) && player(sp->caster)->active());
      DoneSpell* sp2 = new DoneSpell;
      *sp2 = *sp;
      saved_spells.Append(sp2);
    }
    goto resolution;

  } else if (m_turn_type == TURN_NORMAL) {
    if (saved_spells.GetCount()) {
      for (DoneSpellList::Node* node = saved_spells.GetFirst() ;
	   node != NULL ; node = node->GetNext()) {
	m_spells.Append(node->GetData());
      }
      saved_spells.Clear();
    }
  }

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    alive_ones[i] = (being(i) && being(i)->is_alive());
  }

  // Now that all questions have been answered, run the turn. First,
  // look for folks with Permanency and Delayed Effect spells that will
  // capture a spell this turn.
  m_spells.Sort(_spell_list_compare);
  for (DoneSpellList::Node* node = m_spells.GetFirst() ; node != NULL ;
       node = node->GetNext()) {
    DoneSpell* sp = node->GetData();
    if (sp->permanent) {
      g_server->send_event(sp->caster, msg_s_made_permanent(sp));

    } else if (sp->delayed) {
      _msg(sp->caster, msg_s_made_delayed(sp), msg_o_made_delayed(player(sp->caster), sp));
      player(sp->caster)->set_banked(sp->id);
    }
  }

  // Then, run all the spell effects, make monsters hit things, and
  // advance disease and poison damage.
  _exec_all_spells();
  _exec_monsters();
  _exec_illness();

  // Pass time on all creatures, and mark who's dead now that wasn't
  // before we started the turn.
  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i) && being(i)->active()) {
      being(i)->time_passes();
      if (being(i)->is_alive() && being(i)->health() <= 0) {
	being(i)->make_dead();
      }
    }
  }

  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) && !player(i)->is_alive() && alive_ones[i] == true) {
      _msg(i, msg_s_has_died(), msg_o_has_died(player(i)));

    } else if (player(i) && player(i)->is_alive() && player(i)->active()) {
      int time = player(i)->duration(D_PERMANENCY);
      if (time) {
	if (time == 1) {
	  _msg(i, msg_s_permanency_done(), msg_o_permanency_done(player(i)));
	}
	player(i)->age_duration(D_PERMANENCY);
      }

      time = player(i)->duration(D_DELAYED_EFFECT);
      if (time) {
	if (time == 1) {
	  _msg(i, msg_s_delayed_effect_done(), msg_o_delayed_effect_done(player(i)));
	}
	player(i)->age_duration(D_DELAYED_EFFECT);
      }
    }
  }

  // Report the death of creatures, too. Remove their corpse if necessary.
  for (unsigned int i = 0 ; i < m_max_creatures ; i++) {
    Creature* c = creature(i);
    if (c && !c->is_alive()) {
      if (alive_ones[i + MAX_PLAYERS] == true) {
	_msg(i + MAX_PLAYERS, msg_s_has_died(), msg_o_has_died(c));
      }
      if (c->has_corpse() == false) {
	remove_creature(i);
      }
    }
  }

  if (is_game_over()) return;
  if (m_turn_type == TURN_NORMAL) m_turn++;

 resolution:

  // What type of turn should the next one be? This bit tests for haste
  // or timestop effects that need to take precedence next turn, and
  // marks all beings as active or inactive accordingly.
  bool was_hasted = (m_turn_type == TURN_HASTED);
  m_turn_type = TURN_NORMAL;
  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i) && being(i)->is_alive() && being(i)->duration(D_TIMESTOP)) {
      being(i)->age_duration(D_TIMESTOP);
      if (being(i)->duration(D_TIMESTOP)) {
	_msg(i, msg_s_start_timestop(), msg_o_start_timestop(being(i)));
	m_turn_type = TURN_TIMESTOP;
      } else {
	_msg(i, msg_s_end_timestop(), msg_o_end_timestop(being(i)));
      }
    }
  }

  if (m_turn_type != TURN_TIMESTOP) {
    for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
      if (being(i) && being(i)->is_alive() && being(i)->duration(D_HASTE) && was_hasted == false) {
	being(i)->age_duration(D_HASTE);
	switch (being(i)->duration(D_HASTE)) {
	case 3:
	  _msg(i, msg_s_start_haste(), msg_o_start_haste(being(i)));
	  m_turn_type = TURN_HASTED;
	  break;

	case 0:
	  _msg(i, msg_s_end_haste(), msg_o_end_haste(being(i)));
	  break;

	default:
	  _msg(i, msg_s_continue_haste(), msg_o_continue_haste(being(i)));
	  m_turn_type = TURN_HASTED;
	  break;
	}
      }
    }
  }

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i)) {
      if (being(i)->is_alive()) {
	switch (m_turn_type) {
	case TURN_NORMAL:
	  being(i)->set_active(true);
	  break;
	case TURN_HASTED:
	  being(i)->set_active(being(i)->duration(D_HASTE));
	  break;
	case TURN_TIMESTOP:
	  being(i)->set_active(being(i)->duration(D_TIMESTOP));
	  break;
	default: FAIL();
	}
      } else {
	being(i)->set_active(false);
      }
    }
  }
}


bool Rules::is_game_over() {
  int live = 0, dead = 0, surr = 0;
  Player* winner = NULL;
  Player* loser = NULL;

  if (m_game_state == GAME_OVER) {
    return true;
  }

  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL) {
      if (player(i)->is_alive() == false) {
	dead++;
      } else if (player(i)->surrendered()) {
	surr++;
	loser = player(i);
	player(i)->make_dead(false);
      } else {
	live++;
	winner = player(i);
      }
    }
  }
  wxASSERT(live + dead + surr == player_count());

  // If the game is over, print the appropriate messages to all players.
  if (live <= 1) {
    if (live == 1) {
      switch (surr) {
      case 0:  g_server->broadcast(msg_a_sole_survivor(winner));           break;
      case 1:  g_server->broadcast(msg_a_win_by_surrender(winner, loser)); break;
      default: g_server->broadcast(msg_a_win_by_surrender(winner));        break;
      }
      g_server->broadcast(msg_a_who_won(winner));
    } else {
      switch (surr) {
      case 0:  g_server->broadcast(msg_a_everyone_dead());    break;
      case 1:  g_server->broadcast(msg_a_nobody_left(loser)); break;
      default:
	if (dead > 0) {
	  g_server->broadcast(msg_a_survivors_surrender());
	} else {
	  g_server->broadcast(msg_a_everyone_surrenders());
	}
	break;
      }
      g_server->broadcast(msg_a_draw_game());
    }
    m_game_state = GAME_OVER;
    g_server->broadcast(Event::GameOver(*this));
  }

  return m_game_state == GAME_OVER;
}


void Rules::_exec_illness() {
  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i) && being(i)->active()) {
      int time = being(i)->duration(D_DISEASE);
      if (time > 0) {
	_msg(i, msg_s_disease_progress(time), msg_o_disease_progress(being(i), time));
	if (time == 1) {
	  being(i)->make_dead();
	}
      }
      
      time = being(i)->duration(D_POISON);
      if (time > 0) {
	_msg(i, msg_s_poison_progress(time), msg_o_poison_progress(being(i), time));
	if (time == 1) {
	  being(i)->make_dead();
	}
      }
    }
  }
}


void Rules::_exec_all_spells() {
  wxString h;

  for (DoneSpellList::Node* node = m_spells.GetFirst() ; node != NULL ; node = node->GetNext()) {
    DoneSpell* sp = node->GetData();
    Player* p = player(sp->caster);
    wxString t, bname = sp->target >= 0 ? being(sp->target)->name() : wxT("nobody in particular");

    switch (sp->id) {
    case SPELL_PLACEHOLDER: break;
    case SPELL_STAB:
      g_server->broadcast(Event::Msg(wxT("%s stabs with %s %s at %s."), C_STR(p->name()),
				     p->his(), which_hand(sp->hand), C_STR(bname)));
      break;

    default:
      if (sp->delayed == 0) {
	if ((spell_list[ sp->id ].flags & SF_NO_TARGET) > 0) {
	  sp->target = sp->caster;
	} else if (bname.Cmp(p->name()) == 0) {
	  t.Printf(wxT(" on %sself"), p->him());
	} else {
	  t.Printf(wxT(" at %s"), C_STR(bname));
	}

	if (sp->hand == BOTH_HANDS) {
	  h.Printf(wxT("with %s"), which_hand(sp->hand));
	} else {
	  h.Printf(wxT("%s %s %s"),
		   sp->hand == DELAYED_EFFECT ? "from" : "with",
		   p->his(), which_hand(sp->hand));
	}
	g_server->broadcast(Event::Msg(wxT("%s casts %s%s %s%s."), C_STR(p->name()),
				       sp->permanent ? "Permanent " : "",
				       spell_list[ sp->id ].name, C_STR(h), C_STR(t)));
      }
    }
  }

  dispel_magic = fire_storm = ice_storm = false;

  for (DoneSpellList::Node* node = m_spells.GetFirst() ; node != NULL ;
       node = node->GetNext()) {
    DoneSpell* sp = node->GetData();
    if (sp->id == SPELL_COUNTER_SPELL_2) {
      sp->id = SPELL_COUNTER_SPELL;
    }

    if (sp->id == SPELL_DISPEL_MAGIC) {
      dispel_magic = true;
      player(sp->caster)->add_zap(SPELL_SHIELD);

    } else if (sp->id == SPELL_COUNTER_SPELL && _check_visibility(sp)) {
      if (sp->target >= 0) {
	being(sp->target)->add_zap(SPELL_SHIELD);
	being(sp->target)->add_zap(SPELL_COUNTER_SPELL);
      }

    } else if (sp->id == SPELL_MAGIC_MIRROR && _check_visibility(sp)) {
      if (sp->target >= 0) {
	being(sp->target)->add_zap(SPELL_MAGIC_MIRROR);
      }
    }
  }

  if (dispel_magic) {
    g_server->broadcast(msg_a_dispel_magic());
  }

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i)) {
      _exec_counters(i);
    }
  }

  for (DoneSpellList::Node* node = m_spells.GetFirst() ; node != NULL ;
       node = node->GetNext()) {
    DoneSpell* sp = node->GetData();
    switch (sp->id) {
    case SPELL_SHIELD:
    case SPELL_REMOVE_ENCHANTMENT:
    case SPELL_RAISE_DEAD:
    case SPELL_CURE_LIGHT_WOUNDS:
    case SPELL_CURE_HEAVY_WOUNDS:
    case SPELL_SUMMON_GOBLIN:
    case SPELL_SUMMON_OGRE:
    case SPELL_SUMMON_TROLL:
    case SPELL_SUMMON_GIANT:
    case SPELL_SUMMON_ELEMENTAL:
    case SPELL_PROTECTION_FROM_EVIL:
    case SPELL_RESIST_HEAT:
    case SPELL_RESIST_COLD:
    case SPELL_INVISIBILITY:
    case SPELL_HASTE:
    case SPELL_TIME_STOP:
    case SPELL_DELAYED_EFFECT:
    case SPELL_PERMANENCY:
      if (dispel_magic) {
	g_server->broadcast(msg_a_dispelled(sp));

      } else if (sp->target < 0 || _check_visibility(sp) == false) {
	// do nothing;

      } else if (being(sp->target)->zaps(SPELL_COUNTER_SPELL)) {
	_msg(sp->target, msg_s_countered(sp), msg_o_countered(being(sp->target), sp));

      } else {
	being(sp->target)->add_zap(sp->id);
      }
      break;

    case SPELL_STAB:
      if (sp->target >= 0 && _check_visibility(sp)) {
	being(sp->target)->add_zap(sp->id);	
      }
      break;

    case SPELL_LIGHTNING_BOLT_2:
      if (player(sp->caster)->has_used_lb2()) {
	_msg(sp->caster, msg_s_lb2_failed(), msg_o_lb2_failed(player(sp->caster)));
	break;
      }
      sp->id = SPELL_LIGHTNING_BOLT;
      player(sp->caster)->set_used_lb2();
      // FALL THROUGH

    case SPELL_MISSILE:
    case SPELL_FINGER_OF_DEATH:
    case SPELL_LIGHTNING_BOLT:
    case SPELL_CAUSE_LIGHT_WOUNDS:
    case SPELL_CAUSE_HEAVY_WOUNDS:
    case SPELL_FIREBALL:
    case SPELL_AMNESIA:
    case SPELL_CONFUSION:
    case SPELL_CHARM_PERSON:
    case SPELL_CHARM_MONSTER:
    case SPELL_PARALYSIS:
    case SPELL_FEAR:
    case SPELL_ANTI_SPELL:
    case SPELL_DISEASE:
    case SPELL_POISON:
    case SPELL_BLINDNESS:
      if (dispel_magic) {
	g_server->broadcast(msg_a_dispelled(sp));

      } else if (sp->target < 0 || _check_visibility(sp) == false) {
	// do nothing;

	// Finger of Death shreds your pathetic Counter-Spell attempt.
      } else if (being(sp->target)->zaps(SPELL_COUNTER_SPELL) &&
		 sp->id != SPELL_FINGER_OF_DEATH) {
	_msg(sp->target, msg_s_countered(sp), msg_o_countered(being(sp->target), sp));

      } else if (being(sp->target)->zaps(SPELL_MAGIC_MIRROR) &&
		 sp->caster != sp->target) {
	if (being(sp->caster)->zaps(SPELL_MAGIC_MIRROR)) {
	  _msg(sp->caster, msg_s_two_mirrors(being(sp->target), sp),
	       msg_o_two_mirrors(player(sp->caster), being(sp->target), sp),
	       sp->target, msg_t_two_mirrors(player(sp->caster), sp));

	} else {
	  _msg(sp->caster, msg_s_mirrored(being(sp->target), sp),
	       msg_o_mirrored(player(sp->caster), being(sp->target), sp),
	       sp->target, msg_t_mirrored(player(sp->caster), sp));
	  int caster = sp->caster;
	  sp->caster = sp->target;
	  sp->target = caster;
	  if (sp->id != SPELL_FINGER_OF_DEATH &&
	      being(sp->target)->zaps(SPELL_COUNTER_SPELL)) {
	    _msg(sp->target, msg_s_bounce_countered(sp),
		 msg_o_bounce_countered(being(sp->target), sp));

	  } else {
	    being(sp->target)->add_zap(sp->id);
	  }
	}
      } else {
	being(sp->target)->add_zap(sp->id);
      }
      break;

    case SPELL_ICE_STORM:
      if (dispel_magic) {
	g_server->broadcast(msg_a_dispelled(sp));
      } else {
	ice_storm = true;
      }
      break;

    case SPELL_FIRE_STORM:
      if (dispel_magic) {
	g_server->broadcast(msg_a_dispelled(sp));
      } else {
	fire_storm = true;
      }
      break;

    case SPELL_COUNTER_SPELL:
    case SPELL_DISPEL_MAGIC:
    case SPELL_MAGIC_MIRROR:
      // already handled by _exec_counters earlier.
      break;

    case SPELL_SURRENDER:
      FAIL_MSG("Surrender should have been taken care of earlier!");
      break;

    case SPELL_PLACEHOLDER:
      break;

    default:  FAIL();
    }
  }

  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) && player(i)->is_alive()) {
      _exec_summons(i);
    }
  }

  _exec_elements();

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i) && being(i)->is_alive()) {
      _exec_cancels(i);
    }
  }

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i) && being(i)->is_alive()) {
      _exec_protects(i);
    }
  }

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i) && being(i)->is_alive()) {
      _exec_enchants(i);
    }
  }

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i) && being(i)->is_alive()) {
      _exec_attacks(i);
    }
  }

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i)) {
      _exec_heals(i);
    }
  }
}


void Rules::_exec_counters(int i) {
  Being* b = being(i);
  wxASSERT(b != NULL);

  if (dispel_magic) {
    if (b->zaps(SPELL_COUNTER_SPELL)) {
      b->set_zap(SPELL_COUNTER_SPELL, 0);
      _msg(i, msg_s_dispel_counter(), msg_o_dispel_counter(b));
    }

    if (b->zaps(SPELL_MAGIC_MIRROR)) {
      b->set_zap(SPELL_MAGIC_MIRROR, 0);
      _msg(i, msg_s_dispel_mirror(), msg_o_dispel_mirror(b));
    }

    if (i >= (int) MAX_PLAYERS) {
      if (b->is_alive()) {
	g_server->broadcast(msg_a_dispel_creature(b));
	creature(i - MAX_PLAYERS)->destroy_corpse();
	b->hurt(999);

      } else if (creature(i - MAX_PLAYERS)->has_corpse()) {
	g_server->broadcast(msg_a_dispel_corpse(b));
	creature(i - MAX_PLAYERS)->destroy_corpse();
      }
    }

  } else {
    if (b->zaps(SPELL_COUNTER_SPELL)) {
      _msg(i, msg_s_got_counter(), msg_o_got_counter(b));
      if (b->zaps(SPELL_MAGIC_MIRROR)) {
	b->set_zap(SPELL_MAGIC_MIRROR, 0);
	_msg(i, msg_s_counter_mirror(), msg_o_counter_mirror(b));
      }
    }

    if (b->zaps(SPELL_MAGIC_MIRROR)) {
      _msg(i, msg_s_got_mirror(), msg_o_got_mirror(b));
    }
  }
}


bool Rules::_check_visibility(DoneSpell* sp) {
  wxASSERT(sp != NULL);

  if (sp->target >= 0 && sp->target != sp->caster &&
      (being(sp->target)->duration(D_INVISIBILITY) ||
       being(sp->caster)->duration(D_BLINDNESS))) {
    _msg(sp->caster, msg_s_visibility_miss(being(sp->target), sp),
	 msg_o_visibility_miss(player(sp->caster), being(sp->target), sp),
	 sp->target, msg_t_visibility_miss(player(sp->caster), sp));
    return false;
  }
  return true;
}

