// R_Creatures.cpp: Rules for handling monsters and their attacks.

#include "Spellcast.h"
#include "Messages.h"

const int GOBLIN_SYLLABLES = 18;
const int OGRE_SYLLABLES = 23;


static wxString _goblin_name() {
  wxString name;
  int syl, old = -1, numsyl = rand() % 3 + 1;
  static char syllables[GOBLIN_SYLLABLES][4] = {
    "sni", "sna", "fer", "fi", "fir", "por", "per", "snu", "al",
    "an", "erl", "lep", "fru", "fri", "ig", "eg", "thi", "tha"
  };

  for (int i = 0 ; i < numsyl ; i++) {
    do { 
      syl = rand() % GOBLIN_SYLLABLES;
    } while (syl == old);
    name += syllables[syl];
    old = syl;
  }

  if (numsyl == 1 || (rand() % 3 == 0 && numsyl < 3)) {
    name += wxT(" ");
    numsyl = rand() % 2 + 1;
    for (int i = 0 ; i < numsyl ; i++) {
      do {
	syl = rand() % GOBLIN_SYLLABLES;
      } while (syl == old);
      name += syllables[syl];
      old = syl;
    }
  }

  for (size_t i = 0 ; i < name.Len() ; i++) {
    if ((i == 0 || name[i-1] == ' ') && islower(name[i])) {
      name[i] = toupper(name[i]);
    }
  }
  return name;
}


static wxString _ogre_name() {
  wxString name;
  int syl, old = -1, numsyl = rand() % 3 + 2;
  static char syllables[OGRE_SYLLABLES][4] = {
    "kur", "ak", "ral", "ki", "rel", "uk", "kor", "kul", "kas", "lok", "luk", "las",
    "mak", "mok", "mas", "mos", "ga", "tha", "gul", "lug", "mag", "mog", "ug"
  };

  for (int i = 0 ; i < numsyl ; i++) {
    do { 
      syl = rand() % OGRE_SYLLABLES;
    } while (syl == old);
    name += syllables[syl];
    old = syl;
  }

  if (rand() % 3 == 0 && numsyl < 4) {
    name += wxT(" ");
    numsyl = rand() % 3 + 1;
    for (int i = 0 ; i < numsyl ; i++) {
      do {
	syl = rand() % OGRE_SYLLABLES;
      } while (syl == old);
      name += syllables[syl];
      old = syl;
    }
  }

  for (size_t i = 0 ; i < name.Len() ; i++) {
    if ((i == 0 || name[i-1] == ' ') && islower(name[i])) {
      name[i] = toupper(name[i]);
    }
  }
  return name;
}


int Rules::_random_being(unsigned int c) {
  unsigned int candidates[ MAX_BEINGS ];
  unsigned int n_targets = 0;

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i) != NULL && being(i) != creature(c) && being(i)->is_alive()) {
      candidates[ n_targets++ ] = i;
    }
  }

  if (n_targets) {
    return candidates[ rand() % n_targets ];
  }
  return -1;
}


void Rules::_create_monster(unsigned int owner, wxUint8 type, wxInt8 target) {
  wxASSERT(owner < MAX_PLAYERS);
  wxASSERT(player(owner) != NULL);
  wxASSERT(type < NUMBER_OF_CREATURE_TYPES);
  wxString name;

  switch (type) {
  case CREATURE_GOBLIN:  name = _goblin_name() + wxT(" the Goblin"); break;
  case CREATURE_OGRE:    name = _ogre_name() + wxT(" the Ogre");     break;
  case CREATURE_TROLL:   name = _ogre_name() + wxT(" the Troll");    break;
  case CREATURE_GIANT:   name = _ogre_name() + wxT(" the Giant");    break;
  case CREATURE_FIRE_EL: name = _ogre_name() + wxT(" the Fiery");    break;
  case CREATURE_ICE_EL:  name = _ogre_name() + wxT(" the Icy");      break;
  default: FAIL();
  }

  Creature* c = new Creature(type, name, owner);
  add_creature(c);
  if (target >= 0) {
    c->set_target(target);
  }
}


void Rules::_exec_monsters() {
  for (unsigned int i = 0 ; i < m_max_creatures ; i++) {
    Creature* c = creature(i);
    if (c != NULL && c->is_alive() && c->active()) {
      if (c->type() == CREATURE_FIRE_EL || c->type() == CREATURE_ICE_EL) {
	for (unsigned int x = 0 ; x < MAX_BEINGS ; x++) {
	  if (being(x) != NULL && being(x)->is_alive()) {
	    _exec_monster_attack(x, i);
	  }
	}

      } else if (c->duration(D_PARALYSIS) > 0) {
	g_server->broadcast(msg_a_monster_paralyzed(c));
	continue;

      } else if (c->duration(D_AMNESIA) > 0) {
	if (c->prev_target() >= 0 && being(c->prev_target()) != NULL &&
	    being(c->prev_target())->is_alive()) {
	  g_server->broadcast(msg_a_monster_amnesia_1(c, being(c->prev_target())));
	  _exec_monster_attack(c->prev_target(), i);
	} else {
	  g_server->broadcast(msg_a_monster_amnesia_2(c));
	}

      } else if (c->duration(D_CONFUSION) > 0) {
	int target = c->mind_detail;
	if (c->duration(D_CONFUSION) != DURATION_PERMANENT) {
	  target = _random_being(i);
	}
	g_server->broadcast(msg_a_monster_confused(c, being(target)));
	_exec_monster_attack(target, i);

      } else if (c->target() >= 0 && being(c->target()) != NULL &&
		 being(c->target())->is_alive()) {
	_exec_monster_attack(c->target(), i);
	if (c->duration(D_HASTE)) {
	  _exec_monster_attack(c->target(), i);
	}
      }
    }
  }
}


void Rules::_exec_monster_attack(int target, int attacker) {
  wxASSERT(target < (int) MAX_BEINGS);
  wxASSERT(attacker >= 0 && attacker < (int) m_max_creatures);

  if (target < 0) return;
  Being* t = being(target);
  Creature* a = creature(attacker);

  if (t == a) {
    if (!(a->type() == CREATURE_FIRE_EL || a->type() == CREATURE_ICE_EL)) {
      g_server->broadcast(msg_a_monster_not_self(a));
    }
    return;
  }

  if (!(a->type() == CREATURE_FIRE_EL || a->type() == CREATURE_ICE_EL) &&
      t->duration(D_INVISIBILITY) > 0) {
    _msg(target, msg_s_monster_cant_see(a), msg_o_monster_cant_see(a, t));
    return;
  }

  if (t->zaps(SPELL_SHIELD)) {
    _msg(target, msg_s_monster_blocked(a), msg_o_monster_blocked(a, t));
    return;
  }

  if (a->type() == CREATURE_FIRE_EL) {
    if (t->duration(D_RESIST_HEAT) > 0 &&
	(m_turn_type != TURN_TIMESTOP || t->duration(D_TIMESTOP))) {
      _msg(target, msg_s_monster_fireblock(a), msg_o_monster_fireblock(a, t));
    } else {
      _msg(target, msg_s_monster_firethrow(a), msg_o_monster_firethrow(a, t));
      t->hurt(3);
    }

  } else if (a->type() == CREATURE_ICE_EL) {
    if (t->duration(D_RESIST_COLD) > 0 &
	(m_turn_type != TURN_TIMESTOP || t->duration(D_TIMESTOP))) {
      _msg(target, msg_s_monster_iceblock(a), msg_o_monster_iceblock(a, t));
    } else {
      _msg(target, msg_s_monster_icethrow(a), msg_o_monster_icethrow(a, t));
      t->hurt(3);
    }

  } else {
    _msg(target, msg_s_monster_attacks(a), msg_o_monster_attacks(a, t));
    switch (a->type()) {
    case CREATURE_GOBLIN:  t->hurt(1);  break;
    case CREATURE_OGRE:    t->hurt(2);  break;
    case CREATURE_TROLL:   t->hurt(3);  break;
    case CREATURE_GIANT:   t->hurt(4);  break;
    default: FAIL();
    }
  }
}

