// GameData.h: a group of classes which represent aspects of the game's state.

#ifndef __GAMEDATA_H__
#define __GAMEDATA_H__

#include "Event.h"
#include <wx/socket.h>


enum gender_list {
  GENDER_MALE,
  GENDER_FEMALE,
  GENDER_NEUTER,
  GENDER_UNCERTAIN,
  NUMBER_OF_GENDERS
};

enum gestures {
  GESTURE_NOTHING,
  GESTURE_PALM,
  GESTURE_DIGIT,
  GESTURE_FINGERS,
  GESTURE_WAVE,
  GESTURE_CLAP,
  GESTURE_SNAP,
  GESTURE_KNIFE,
  GESTURE_UNKNOWN,
  GESTURE_ANTISPELL,
  NUMBER_OF_GESTURES,

  MAX_GESTURES = 8,
  NUMBER_OF_PLAYER_GESTURES = 8
};

enum hands {
  LEFT_HAND,
  RIGHT_HAND,
  NUMBER_OF_HANDS,
  NEITHER_HAND,
  BOTH_HANDS,
  DELAYED_EFFECT
};

enum game_states {
  NO_GAME_YET,
  GAME_NOT_STARTED,
  GAME_IN_PROGRESS,
  GAME_OVER,
  NUMBER_OF_GAME_STATES
};

enum turn_phases {
  NO_PHASE,
  GESTURES_PHASE,
  ANSWERS_PHASE,
  NUMBER_OF_TURN_PHASES
};

enum creature_types {
  CREATURE_GOBLIN = 1,
  CREATURE_OGRE,
  CREATURE_TROLL,
  CREATURE_GIANT,
  CREATURE_ICE_EL,
  CREATURE_FIRE_EL,
  NUMBER_OF_CREATURE_TYPES,

  DEFAULT_MAX_CREATURES = 64
};

enum question_types {
  QUESTION_SPELL,
  QUESTION_TARGET,
  QUESTION_NEW_MONSTER,
  QUESTION_MONSTER,
  QUESTION_ELEMENTAL,
  QUESTION_USE_DELAYED,
  QUESTION_PERMANENT,
  QUESTION_DELAYED,
  QUESTION_HAND,
  QUESTION_CHARM,
  NUMBER_OF_QUESTION_TYPES,

  MAX_QUESTIONS = 4
};

enum spell_list {
  // Protection
  SPELL_SHIELD,
  SPELL_REMOVE_ENCHANTMENT,
  SPELL_MAGIC_MIRROR,
  SPELL_COUNTER_SPELL,
  SPELL_COUNTER_SPELL_2,
  SPELL_DISPEL_MAGIC,
  SPELL_RAISE_DEAD,
  SPELL_CURE_LIGHT_WOUNDS,
  SPELL_CURE_HEAVY_WOUNDS,

  // Summoning
  SPELL_SUMMON_GOBLIN,
  SPELL_SUMMON_OGRE,
  SPELL_SUMMON_TROLL,
  SPELL_SUMMON_GIANT,
  SPELL_SUMMON_ELEMENTAL,

  // Damaging
  SPELL_MISSILE,
  SPELL_FINGER_OF_DEATH,
  SPELL_LIGHTNING_BOLT,
  SPELL_LIGHTNING_BOLT_2,
  SPELL_CAUSE_LIGHT_WOUNDS,
  SPELL_CAUSE_HEAVY_WOUNDS,
  SPELL_FIREBALL,
  SPELL_FIRE_STORM,
  SPELL_ICE_STORM,
  
  // Enchantments
  SPELL_AMNESIA,             // (mind)
  SPELL_CONFUSION,
  SPELL_CHARM_PERSON,
  SPELL_CHARM_MONSTER,
  SPELL_PARALYSIS,
  SPELL_FEAR,

  SPELL_ANTI_SPELL,          // (regular)
  SPELL_PROTECTION_FROM_EVIL,
  SPELL_RESIST_HEAT,
  SPELL_RESIST_COLD,
  SPELL_DISEASE,
  SPELL_POISON,
  SPELL_BLINDNESS,
  SPELL_INVISIBILITY,
  SPELL_HASTE,
  SPELL_TIME_STOP,
  SPELL_DELAYED_EFFECT,
  SPELL_PERMANENCY,

  // Not actually spells
  SPELL_SURRENDER,
  SPELL_STAB,
  SPELL_PLACEHOLDER,    // an invalid "dummy" spell id

  NUMBER_OF_SPELLS,
  NUMBER_OF_MIND_SPELLS = 6
};

enum spell_flags {
  SF_PROTECTION     = 0x0001,
  SF_SUMMONING      = 0x0002,
  SF_DAMAGING       = 0x0004,
  SF_ENCHANTMENT    = 0x0008,
  SF_MIND           = 0x0010,
  SF_NOT_SPELL      = 0x0020,
  SF_NO_TARGET      = 0x0040,
  SF_SELF           = 0x0080,
  SF_MONSTER        = 0x0100,
  SF_NOPERM         = 0x0200,
};


struct Spell {
  wxUint8 id;
  const char* name;
  const char* mainhand;
  const char* offhand;
  wxUint8 duration;
  wxUint32 flags;
};

struct PossibleSpell {
  wxUint8 id;
  wxUint8 hand;
};

struct DoneSpell {
  wxInt8 id;
  wxInt8 hand;
  wxInt8 target;
  wxInt8 caster;
  wxUint8 mind_hand;
  wxInt8 detail;
  wxUint8 n_questions;
  wxInt8 permanent;
  wxInt8 delayed;
};

WX_DECLARE_LIST(PossibleSpell, PossibleSpellList);


enum duration_entries {
  NO_DURATION,
  D_AMNESIA,
  D_CONFUSION,
  D_CHARM_PERSON,
  D_CHARM_MONSTER,
  D_PARALYSIS,
  D_FEAR,
  D_RESIST_HEAT,
  D_RESIST_COLD,
  D_PROTECTION,
  D_HASTE,
  D_TIMESTOP,
  D_INVISIBILITY,
  D_BLINDNESS,
  D_POISON,
  D_DISEASE,
  D_RESIST_ICESTORM,
  D_DELAYED_EFFECT,
  D_PERMANENCY,
  NUMBER_OF_DURATION_ENTRIES,

  DURATION_PERMANENT = 222,
};


class Being {
 public:
  Being(const wxString& name, wxUint8 gender, int max_hp);
  Being(MemBuf* data);
  virtual inline ~Being() {}

  virtual MemBuf serialize() const;
  inline const wxString& name() const { return m_name; }
  inline bool is_alive() const { return m_alive == 1; }
  inline void make_dead(bool rez = true) { m_alive = 0; m_raiseable = rez; }
  inline void make_alive() { wxASSERT(m_raiseable); m_alive = 1; }
  inline int raiseable() const { return m_raiseable; }
  inline void hurt(int hp) { m_health -= hp; }
  inline void heal(int hp) { m_health += hp; }
  inline int health() const { return m_health; }
  inline int max_health() const { return m_max_health; }
  inline bool active() const { return m_active != 0; }
  inline void set_active(bool yesno) { m_active = yesno; }
  inline wxUint8 duration(wxUint8 i) const { return m_durations[i]; }
  void set_duration(wxUint8 i, wxUint8 t);
  void age_duration(wxUint8 i);
  void time_passes();
  inline void add_zap(wxUint8 spell) { m_zaplist[spell]++; }
  inline void set_zap(wxUint8 spell, wxUint8 i) { m_zaplist[spell] = i; }
  inline wxUint8 zaps(wxUint8 spell) { return m_zaplist[spell]; }
  inline bool ice_storm_immune() const { return m_ice_storm_immune; }
  inline void set_ice_storm_immune() { m_ice_storm_immune = true; }
  inline bool asked() const { return m_asked; }
  inline void set_asked() { m_asked = true; }
  wxString status() const;

  const char* he(bool caps = false) const;
  const char* him(bool caps = false) const;
  const char* his(bool caps = false) const;
  inline const char* He() const { return he(true); }
  inline const char* Him() const { return him(true); }
  inline const char* His() const { return his(true); }

  wxUint8  mind_hand;
  wxUint8  mind_detail;

 protected:
  wxString m_name;
  wxUint8  m_gender;
  wxUint8  m_alive;
  wxInt16  m_health;
  wxInt16  m_max_health;
  wxInt8   m_active;
  wxUint8  m_durations[ NUMBER_OF_DURATION_ENTRIES ];
  wxUint8  m_zaplist[ NUMBER_OF_SPELLS ];
  bool     m_ice_storm_immune;
  bool     m_raiseable;
  bool     m_asked;
};


class Creature : public Being {
 public:
  Creature(wxUint8 type, const wxString& name, wxInt8 owner);
  Creature(MemBuf* data);

  MemBuf serialize() const;
  inline wxUint8 type() const { return m_type; }
  inline wxInt8 owner() const { return m_owner; }
  inline void set_owner(wxInt8 owner)
    { wxASSERT(owner < (int) MAX_PLAYERS); m_owner = owner; }
  inline wxInt8 target() const { return m_target; }
  void set_target(wxInt8 target);
  inline wxInt8 prev_target() const { return m_prev_target; }
  inline bool has_corpse() const { return m_corpseless == 0; }
  inline void destroy_corpse() { m_corpseless = 1; }
  inline void undestroy_corpse() { m_corpseless = 0; }

 private:
  wxUint8 m_type;
  wxInt8  m_owner;
  wxInt8  m_target;
  wxInt8  m_prev_target;
  wxUint8 m_corpseless;
};


class Player : public Being {
 public:
  Player(const wxString& name, wxUint8 gender);
  Player(Event& name_event);
  Player(MemBuf* data);

  MemBuf serialize() const;
  bool surrendered(bool set = false);
  inline bool done() const { return m_done != 0; }
  inline void set_done(bool yesno) { m_done = yesno; }
  inline bool is_me() const { return m_me != 0; }
  inline void this_is_me() { m_me = 1; }
  inline wxInt8 banked() const { return m_banked; }
  inline void set_banked(wxInt8 spell) { m_banked = spell; }
  wxUint8 get_gesture(wxUint8 hand, wxUint8 which) const;
  void set_gesture(wxUint8 hand, wxUint8 gesture);
  void add_gestures(const Event& hands_event);
  void scroll_up();
  void clear_gestures();
  void use_default_gestures();
  PossibleSpellList* get_spells();
  bool count_spell_questions() const { return m_spell_questions; }
  void add_spell_question() { m_spell_questions++; }
  void remove_spell_question() { m_spell_questions--; }
  bool asked_perm() const { return m_asked_perm; }
  void set_asked_perm(bool yesno) { m_asked_perm = yesno; }
  bool asked_delay() const { return m_asked_delay; }
  void set_asked_delay(bool yesno) { m_asked_delay = yesno; }
  bool asked_banked() const { return m_asked_banked; }
  void set_asked_banked(bool yesno) { m_asked_banked = yesno; }
  bool has_used_lb2() const { return m_lb2; }
  void set_used_lb2() { m_lb2 = true; }

 private:
  wxUint8 m_gestures[ NUMBER_OF_HANDS ][ MAX_GESTURES ];
  wxUint8 m_surrendered;
  wxInt8  m_banked;
  wxInt8  m_done;
  wxInt8  m_me;
  wxUint8 m_spell_questions;
  bool    m_asked_perm;
  bool    m_asked_delay;
  bool    m_asked_banked;
  bool    m_lb2;
};


class Question {
 public:
  Question(const wxString& question);
  Question(const wxString& question, int n_answers, const wxString msgs[]);
  Question(const Question& other);
  void operator=(const Question& other);
  ~Question();

  inline int type() const { return m_type; }
  inline void set_type(int type) { m_type = type; }
  inline wxInt8 spell() const { return m_spell; }
  inline void set_spell(wxInt8 sp) { m_spell = sp; }
  inline wxInt8 monster() const { return m_monster; }
  inline void set_monster(wxInt8 i) { m_monster = i; }
  inline bool has_been_sent() const { return m_sent; }
  inline void set_sent() { m_sent = true; }
  inline const wxString& question() const { return m_question; }
  inline int answer_count() const { return m_ans_count; }
  inline const wxString* answers() const { return m_answers; }
  inline const wxString& answer(unsigned int i) const
    { wxASSERT((int) i < m_ans_count); return m_answers[i]; }
  inline Event to_event() const
    { return Event::Question(question(), answer_count(), answers()); }
  void add_answer(const wxString& answer);
  wxString to_s() const;

 private:
  wxString  m_question;
  wxString* m_answers;
  int       m_ans_count;
  int       m_type;
  wxInt8    m_spell;
  wxInt8    m_monster;
  bool      m_sent;
};


char gesture_to_char(wxUint8 gesture);
const char* your_hand(int hand);
const char* which_hand(int hand);
const char* gesture_name(int gesture);

extern struct Spell spell_list[ NUMBER_OF_SPELLS ];


#endif   // __GAMEDATA_H__
