// GameData.cpp: a group of classes which represent aspects of the game's state.

#include "Spellcast.h"
#include <wx/listimpl.cpp>
WX_DEFINE_LIST(PossibleSpellList);
WX_DEFINE_LIST(DoneSpellList);


struct Spell spell_list[ NUMBER_OF_SPELLS ] = {
  { SPELL_SHIELD, "Shield", "P", "", 1, SF_PROTECTION | SF_SELF },
  { SPELL_REMOVE_ENCHANTMENT, "Remove Enchantment", "PWDP", "", 0, SF_PROTECTION | SF_SELF },
  { SPELL_MAGIC_MIRROR, "Magic Mirror", "WC", "WC", 1, SF_PROTECTION | SF_SELF},
  { SPELL_COUNTER_SPELL, "Counter-Spell", "PPW", "", 1, SF_PROTECTION | SF_SELF },
  { SPELL_COUNTER_SPELL_2, "Counter-Spell", "SWW", "", 1, SF_PROTECTION | SF_SELF },
  { SPELL_DISPEL_MAGIC, "Dispel Magic", "WPDC", "???C", 1, SF_PROTECTION | SF_NO_TARGET },
  { SPELL_RAISE_DEAD, "Raise Dead", "CWFWWD", "C", 1, SF_PROTECTION },
  { SPELL_CURE_LIGHT_WOUNDS, "Cure Light Wounds", "WFD", "", 0, SF_PROTECTION | SF_SELF },
  { SPELL_CURE_HEAVY_WOUNDS, "Cure Heavy Wounds", "WPFD", "", 0, SF_PROTECTION | SF_SELF },

  { SPELL_SUMMON_GOBLIN, "Summon Goblin", "WFS", "", 0, SF_SUMMONING | SF_NO_TARGET | SF_MONSTER },
  { SPELL_SUMMON_OGRE, "Summon Ogre", "WFSP", "", 0, SF_SUMMONING | SF_NO_TARGET | SF_MONSTER },
  { SPELL_SUMMON_TROLL, "Summon Troll", "WFSPF", "", 0, SF_SUMMONING | SF_NO_TARGET | SF_MONSTER },
  { SPELL_SUMMON_GIANT, "Summon Giant", "WFSPFW", "", 0, SF_SUMMONING | SF_NO_TARGET | SF_MONSTER },
  { SPELL_SUMMON_ELEMENTAL, "Summon Elemental", "SWWSC", "????C", 0, SF_SUMMONING | SF_NO_TARGET },

  { SPELL_MISSILE, "Missile", "DS", "", 0, SF_DAMAGING },
  { SPELL_FINGER_OF_DEATH, "Finger of Death", "DSSSFPWP", "", 0, SF_DAMAGING },
  { SPELL_LIGHTNING_BOLT, "Lightning Bolt", "DDFFD", "", 0, SF_DAMAGING },
  { SPELL_LIGHTNING_BOLT_2, "Lightning Bolt", "CDDW", "C", 0, SF_DAMAGING },
  { SPELL_CAUSE_LIGHT_WOUNDS, "Cause Light Wounds", "PFW", "", 0, SF_DAMAGING },
  { SPELL_CAUSE_HEAVY_WOUNDS, "Cause Heavy Wounds", "DFPW", "", 0, SF_DAMAGING },
  { SPELL_FIREBALL, "Fireball", "DDSSF", "", 0, SF_DAMAGING },
  { SPELL_FIRE_STORM, "Fire Storm", "CWWS", "C", 0, SF_DAMAGING | SF_NO_TARGET },
  { SPELL_ICE_STORM, "Ice Storm", "CSSW", "C", 0, SF_DAMAGING | SF_NO_TARGET },

  { SPELL_AMNESIA, "Amnesia", "PPD", "", 2, SF_ENCHANTMENT | SF_MIND },
  { SPELL_CONFUSION, "Confusion", "FSD", "", 2, SF_ENCHANTMENT | SF_MIND },
  { SPELL_CHARM_PERSON, "Charm Person", "FDSP", "", 2, SF_ENCHANTMENT | SF_MIND },
  { SPELL_CHARM_MONSTER, "Charm Monster", "DDSP", "", 1, SF_ENCHANTMENT | SF_MIND },
  { SPELL_PARALYSIS, "Paralysis", "FFF", "", 2, SF_ENCHANTMENT | SF_MIND },
  { SPELL_FEAR, "Fear", "DWS", "", 2, SF_ENCHANTMENT | SF_MIND },
  { SPELL_ANTI_SPELL, "Anti-Spell", "FPS", "", 0, SF_ENCHANTMENT | SF_NOPERM },
  { SPELL_PROTECTION_FROM_EVIL, "Protection From Evil", "PWW", "", 4, SF_ENCHANTMENT | SF_SELF },
  { SPELL_RESIST_HEAT, "Resist Heat", "PFWW", "", DURATION_PERMANENT, SF_ENCHANTMENT | SF_SELF },
  { SPELL_RESIST_COLD, "Resist Cold", "PFSS", "", DURATION_PERMANENT, SF_ENCHANTMENT | SF_SELF },
  { SPELL_DISEASE, "Disease", "CFFFSD", "C", 6, SF_ENCHANTMENT | SF_NOPERM },
  { SPELL_POISON, "Poison", "DWFWWD", "", 6, SF_ENCHANTMENT | SF_NOPERM },
  { SPELL_BLINDNESS, "Blindness", "DFFWD", "D", 4, SF_ENCHANTMENT },
  { SPELL_INVISIBILITY, "Invisibility", "SWPP", "SW", 4, SF_ENCHANTMENT | SF_SELF },
  { SPELL_HASTE, "Haste", "CWWPWP", "C", 4, SF_ENCHANTMENT | SF_SELF },
  { SPELL_TIME_STOP, "Time Stop", "CPPS", "C", 2, SF_ENCHANTMENT | SF_SELF | SF_NOPERM },
  { SPELL_DELAYED_EFFECT, "Delayed Effect", "PSSSWD", "", 4, SF_ENCHANTMENT | SF_NO_TARGET | SF_NOPERM },
  { SPELL_PERMANENCY, "Permanency", "WDSPFPS", "", 4, SF_ENCHANTMENT | SF_NO_TARGET | SF_NOPERM },

  { SPELL_SURRENDER, "Surrender", "P", "P", 0, SF_NOT_SPELL | SF_NO_TARGET },
  { SPELL_STAB, "Stab", "K", "", 0, SF_NOT_SPELL },
  { SPELL_PLACEHOLDER, "EVERYTHING IS RUINED FOREVER", "WTF", "WTF", 0, SF_NOT_SPELL | SF_NO_TARGET },
};


Being::Being(const wxString& name, wxUint8 gender, int max_hp) {
  wxASSERT(name.Len() > 0);
  wxASSERT(gender < NUMBER_OF_GENDERS);
  wxASSERT(max_hp > 0);

  m_name = name;
  m_gender = gender;
  m_alive = 1;
  m_health = max_hp;
  m_max_health = max_hp;
  m_active = 1;
  mind_hand = 0;
  mind_detail = 0;
  for (int i = 0 ; i < NUMBER_OF_DURATION_ENTRIES ; i++) {
    m_durations[i] = 0;
  }
  for (int i = 0 ; i < NUMBER_OF_SPELLS ; i++) {
    m_zaplist[i] = 0;
  }
  m_raiseable = m_ice_storm_immune = m_asked = false;
}


Being::Being(MemBuf* data) {
  if (unserialize(data, "TBBssbBB", &m_name, &m_gender, &m_alive, &m_health, &m_max_health,
		   &m_active, &mind_hand, &mind_detail) == 0) {
    FAIL_MSG("Bogus Being unserialize!");
  }
  for (int i = 0 ; i < NUMBER_OF_DURATION_ENTRIES ; i++) {
    if (unserialize(data, "B", &m_durations[i]) == 0) {
      FAIL_MSG("Bogus Being unserialize!");
    }
  }
  for (int i = 0 ; i < NUMBER_OF_SPELLS ; i++) {
    m_zaplist[i] = 0;
  }
  m_raiseable = m_ice_storm_immune = m_asked = false;
}


MemBuf Being::serialize() const {
  MemBuf output = ::serialize("TBBssbBB", &m_name, m_gender, m_alive, m_health, m_max_health,
			     m_active, mind_hand, mind_detail);
  for (int i = 0 ; i < NUMBER_OF_DURATION_ENTRIES ; i++) {
    output.append(&m_durations[i], 1);
  }
  return output;
}


void Being::set_duration(wxUint8 i, wxUint8 t) {
  wxASSERT(i < NUMBER_OF_DURATION_ENTRIES);
  if (i != NO_DURATION) {
    m_durations[i] = t;
  }
}


void Being::age_duration(wxUint8 i) {
  wxASSERT(i < NUMBER_OF_DURATION_ENTRIES);
  if (m_durations[i] > 0 && m_durations[i] != DURATION_PERMANENT) {
    m_durations[i]--;
  }
}


void Being::time_passes() {
  for (int i = 1 ; i < NUMBER_OF_DURATION_ENTRIES ; i++) {
    switch (i) {
    case D_DELAYED_EFFECT:
    case D_PERMANENCY:
    case D_TIMESTOP:
    case D_HASTE:
      break;
    default: age_duration(i);
    }
  }
  for (int i = 0 ; i < NUMBER_OF_SPELLS ; i++) {
    m_zaplist[i] = 0;
  }
  if (m_durations[D_AMNESIA] + m_durations[D_CONFUSION] + m_durations[D_CHARM_PERSON] +
      m_durations[D_PARALYSIS] + m_durations[D_FEAR] == 0) {
    mind_hand = mind_detail = 0;
  }
  m_ice_storm_immune = m_asked = false;
}


wxString Being::status() const {
  wxString s;
  if (duration(D_INVISIBILITY)) { s += wxT('I'); }
  if (duration(D_BLINDNESS))    { s += wxT('b'); }
  if (duration(D_PROTECTION))   { s += wxT('P'); }
  if (duration(D_RESIST_HEAT))  { s += wxT('H'); }
  if (duration(D_RESIST_COLD))  { s += wxT('C'); }
  if (duration(D_DISEASE))      { s += wxT('d'); }
  if (duration(D_POISON))       { s += wxT('p'); }
  return s;
}


const char* Being::he(bool caps) const {
  static const char* _spn[ NUMBER_OF_GENDERS * 2 ] = {
    "he", "she", "it", "ke", "He", "She", "It", "Ke"
  };
  wxASSERT(m_gender < NUMBER_OF_GENDERS);
  return _spn[ m_gender + (caps ? NUMBER_OF_GENDERS : 0) ];
}
 

const char* Being::him(bool caps) const {
  static const char* _opn[ NUMBER_OF_GENDERS * 2 ] = {
    "him", "her", "it", "hir", "Him", "Her", "It", "Hir"
  };
  wxASSERT(m_gender < NUMBER_OF_GENDERS);
  return _opn[ m_gender + (caps ? NUMBER_OF_GENDERS : 0) ];
}


const char* Being::his(bool caps) const {
  static const char* _ppn[ NUMBER_OF_GENDERS * 2 ] = {
    "his", "her", "its", "hir", "His", "Her", "Its", "Hir"
  };
  wxASSERT(m_gender < NUMBER_OF_GENDERS);
  return _ppn[ m_gender + (caps ? NUMBER_OF_GENDERS : 0) ];
}


Creature::Creature(wxUint8 type, const wxString& name, wxInt8 owner)
  : Being(name, GENDER_UNCERTAIN, 999) {
  m_gender = GENDER_MALE;
  switch (type) {
  case CREATURE_GOBLIN:   m_max_health = m_health = 1;    break;
  case CREATURE_OGRE :    m_max_health = m_health = 2;    break;
  case CREATURE_TROLL:    m_max_health = m_health = 3;    break;
  case CREATURE_GIANT:    m_max_health = m_health = 4;    break;
  case CREATURE_FIRE_EL:
  case CREATURE_ICE_EL:
    m_max_health = m_health = 3;
    m_gender = GENDER_NEUTER;
    break;
  default:  FAIL();
  }
  m_type = type;
  m_owner = owner;
  m_target = m_prev_target = -1;
  m_corpseless = 0;
}


Creature::Creature(MemBuf* data) : Being(data) {
  if (unserialize(data, "BbB", &m_type, &m_owner, &m_corpseless) == 0) {
    FAIL_MSG("Bogus Creature unserialize!");
  }
  m_target = m_prev_target = -1;
}


MemBuf Creature::serialize() const {
  MemBuf output = this->Being::serialize();
  output.append(::serialize("BbB", m_type, m_owner, m_corpseless));
  return output;
}


void Creature::set_target(wxInt8 target) {
  wxASSERT(target >= -1);
  m_prev_target = m_target;
  m_target = target;
}


Player::Player(const wxString& name, wxUint8 gender) : Being(name, gender, 15) {
  m_surrendered = m_done = m_me = 0;
  m_lb2 = m_asked_perm = m_asked_delay = m_asked_banked = false;
  m_spell_questions = 0;
  m_banked = -1;
  clear_gestures();
}


Player::Player(Event& name_event) : Being(wxT("honk"), GENDER_MALE, 15) {
  wxASSERT(name_event.type() == EVENT_NAME);
  m_gender = name_event.arg1();
  m_name = name_event.data()->as_str();
  m_surrendered = m_done = m_me = 0;
  m_lb2 = m_asked_perm = m_asked_delay = m_asked_banked = false;
  m_spell_questions = 0;
  m_banked = -1;
  clear_gestures();
}


Player::Player(MemBuf* data) : Being(data) {
  if (unserialize(data, "B", &m_surrendered) == 0) {
    FAIL_MSG("Bogus Player unserialize");
  }
  m_lb2 = m_asked_perm = m_asked_delay = m_asked_banked = false;
  m_spell_questions = 0;
  m_done = m_me = 0;
  m_banked = -1;
}


MemBuf Player::serialize() const {
  MemBuf output = this->Being::serialize();
  output.append(::serialize("B", m_surrendered));
  return output;
}


bool Player::surrendered(bool set) {
  if (set || m_surrendered > 0) {
    m_surrendered = 1;
    return true;
  }
  return false;
}


wxUint8 Player::get_gesture(wxUint8 hand, wxUint8 which) const {
  wxASSERT(hand < NUMBER_OF_HANDS);
  wxASSERT(which < MAX_GESTURES);
  return m_gestures[ hand ][ which ];
}


void Player::set_gesture(wxUint8 hand, wxUint8 gesture) {
  wxASSERT(hand < NUMBER_OF_HANDS);
  wxASSERT(gesture < NUMBER_OF_GESTURES);
  m_gestures[ hand ][ 0 ] = gesture;
}


void Player::add_gestures(const Event& hands_event) {
  wxASSERT(hands_event.type() == EVENT_SV_HANDS);
  scroll_up();
  m_gestures[ LEFT_HAND ][0]  = hands_event.arg2();
  m_gestures[ RIGHT_HAND ][0] = hands_event.arg3();
}


void Player::scroll_up() {
  for (int i = MAX_GESTURES - 1 ; i > 0 ; i--) {
    m_gestures[ LEFT_HAND ][ i ]  = m_gestures[ LEFT_HAND ][ i - 1 ];
    m_gestures[ RIGHT_HAND ][ i ] = m_gestures[ RIGHT_HAND ][ i - 1 ];
  }
  m_gestures[ LEFT_HAND ][0] = m_gestures[ RIGHT_HAND ][0] = GESTURE_NOTHING;
}


void Player::clear_gestures() {
  for (int i = 0 ; i < NUMBER_OF_HANDS ; i++) {
    for (int j = 0 ; j < MAX_GESTURES ; j++) {
      m_gestures[i][j] = GESTURE_NOTHING;
    }
  }
}


static int char_to_gesture(wxUint8 gesture) {
  switch (gesture) {
  case '_':  return GESTURE_NOTHING;
  case 'P':  return GESTURE_PALM;
  case 'D':  return GESTURE_DIGIT;
  case 'F':  return GESTURE_FINGERS;
  case 'K':  return GESTURE_KNIFE;
  case 'W':  return GESTURE_WAVE;
  case 'C':  return GESTURE_CLAP;
  case 'S':  return GESTURE_SNAP;
  case 'U':  return GESTURE_UNKNOWN;
  case 'Z':  return GESTURE_ANTISPELL;
  default: FAIL();
  }
  return 'X';
}


// For debugging, the server player can specify a starting hand state
// for all players -- it won't show up on the client's displays, but
// it's there.
void Player::use_default_gestures() {
  for (int hand = 0 ; hand < NUMBER_OF_HANDS ; hand++) {
    wxString s = StringReverse(g_initial[hand]);
    for (size_t i = 0 ; i < s.length() ; i++) {
      s[i] = char_to_gesture(s[i]);
    }
    memmove(m_gestures[hand], C_STR(s),
	    (s.length() > MAX_GESTURES) ? MAX_GESTURES : s.length());
  }
}


char gesture_to_char(wxUint8 gesture) {
  switch (gesture) {
  case GESTURE_NOTHING:    return '_';
  case GESTURE_PALM:       return 'P';
  case GESTURE_DIGIT:      return 'D';
  case GESTURE_FINGERS:    return 'F';
  case GESTURE_KNIFE:      return 'K';
  case GESTURE_WAVE:       return 'W';
  case GESTURE_CLAP:       return 'C';
  case GESTURE_SNAP:       return 'S';
  case GESTURE_UNKNOWN:    return 'U';
  case GESTURE_ANTISPELL:  return 'Z';
  default: FAIL();
  }
  return 'X';
}


const char* your_hand(int hand) {
  switch (hand) {
  case LEFT_HAND:       return "your left hand";
  case RIGHT_HAND:      return "your right hand";
  case BOTH_HANDS:      return "both hands";
  case DELAYED_EFFECT:  return "your Delayed Effect";
  default: FAIL();
  }
  return NULL;
}


const char* which_hand(int hand) {
  switch (hand) {
  case LEFT_HAND:       return "left hand";
  case RIGHT_HAND:      return "right hand";
  case BOTH_HANDS:      return "both hands";
  case DELAYED_EFFECT:  return "Delayed Effect";
  default: FAIL();
  }
  return NULL;
}


const char* gesture_name(int gesture) {
  switch (gesture) {
  case GESTURE_NOTHING:   return "nothing";
  case GESTURE_PALM:      return "palm";
  case GESTURE_DIGIT:     return "digit";
  case GESTURE_FINGERS:   return "fingers";
  case GESTURE_KNIFE:     return "stab";
  case GESTURE_WAVE:      return "wave";
  case GESTURE_CLAP:      return "clap";
  case GESTURE_SNAP:      return "snap";
  case GESTURE_UNKNOWN:   return "UNKNOWN";
  case GESTURE_ANTISPELL: return "ANTISPELL";
  default: FAIL();
  }
  return "ITCHY SPLEEN";
}


static bool compare(const char* spell, const wxString& hand) {
  for (size_t i = 0 ; spell[i] ; i++) {
    if (hand[i] != spell[i] && spell[i] != '?') {
      return false;
    }
  }
  return true;
}


PossibleSpellList* Player::get_spells() {
  PossibleSpell* spell = NULL;
  PossibleSpellList* list = new PossibleSpellList();
  list->DeleteContents(true);

  wxString left, right;
  for (int i = 0 ; i < MAX_GESTURES ; i++) {
    left  += gesture_to_char(m_gestures[ LEFT_HAND ][ i ]);
    right += gesture_to_char(m_gestures[ RIGHT_HAND ][ i ]);
  }
  fprintf(stderr, "Left: %s    Right: %s\n", C_STR(left), C_STR(right));

  for (int i = 0 ; i < NUMBER_OF_SPELLS ; i++) {
    Spell sp = spell_list[i];
    if (compare(sp.mainhand, left) && compare(sp.offhand, right)) {
      if (sp.id == SPELL_SURRENDER) {
	surrendered(true);
      } else {
	spell = new PossibleSpell;
	spell->id = i;
	spell->hand = (sp.offhand[0] != '\0' && sp.offhand[0] != '?') ?
	  BOTH_HANDS : LEFT_HAND;
	list->Append(spell);
	if (spell->hand == BOTH_HANDS) {
	  continue;
	}
      }
    }

    if (compare(sp.mainhand, right) && compare(sp.offhand, left)) {
      if (sp.id == SPELL_SURRENDER) {
	surrendered(true);
      } else {
	spell = new PossibleSpell;
	spell->id = i;
	spell->hand = (sp.offhand[0] != '\0' && sp.offhand[0] != '?') ?
	  BOTH_HANDS : RIGHT_HAND;
	list->Append(spell);
      }
    }
  }

  return list;
}


Question::Question(const wxString& question) {
  int i;

  m_question = (wxString) question;
  m_ans_count = 0;
  m_answers = NULL;
  m_type = m_spell = m_monster = -1;
  m_sent = false;

  // is it a null-separated Event::Question data string? if so, parse it.
  if ((i = StringFind(question, '\0')) >= 0) {
    wxString foo(question);
    m_question = foo.Left(i);
    StringRemove(foo, 0, i + 1);

    while ((i = StringFind(foo, '\0')) >= 0) {
      add_answer(foo.Left(i));
      StringRemove(foo, 0, i + 1);
    }
    add_answer(foo);
  }
}


Question::Question(const wxString& question, int n_answers, const wxString msgs[]) {
  m_question = (wxString) question;
  m_ans_count = n_answers;
  m_type = m_spell = m_monster = -1;
  m_sent = false;
  m_answers = new wxString[ m_ans_count ];
  for (int i = 0 ; i < n_answers ; i++) {
    m_answers[i] = msgs[i];
  }
}


Question::Question(const Question& other) {
  m_question = other.question();
  m_ans_count = other.answer_count();
  m_type = other.type();
  m_spell = other.spell();
  m_monster = other.monster();
  m_sent = other.has_been_sent();
  m_answers = new wxString[ m_ans_count ];
  for (int i = 0 ; i < m_ans_count ; i++) {
    m_answers[i] = other.answer(i);
  }
}


void Question::operator=(const Question& other) {
  if (this != &other) {
    delete [] m_answers;
    m_answers = new wxString[ other.answer_count() ];
    for (int i = 0 ; i < other.answer_count() ; i++) {
      m_answers[i] = other.answer(i);
    }
  }
}


Question::~Question() {
  if (m_answers != NULL) {
    delete [] m_answers;
  }
}


void Question::add_answer(const wxString& answer) {
  wxString* oldanswers = m_answers;
  m_answers = new wxString[ ++m_ans_count ];
  if (oldanswers != NULL) {
    for (int i = 0 ; i < m_ans_count - 1 ; i++) {
      m_answers[i] = oldanswers[i];
    }
    delete [] oldanswers;
  }
  m_answers[ m_ans_count - 1 ] = (wxString) answer;
}


wxString Question::to_s() const {
  wxString s;
  s.Printf("QUESTION about spell %d, %d answers\n  \"%s\"\n",
	   m_spell, m_ans_count, m_question.c_str());
  for (int i = 0 ; i < m_ans_count ; i++) {
    wxString x;
    x.Printf("    %d. %s\n", i, m_answers[i].c_str());
    s += x;
  }
  return s;
}
