/* $Id: Client.cpp,v 1.14 2008/05/04 09:19:51 dpt Exp $
 *
 * Client.cpp: a class that controls interaction between the user and server.
 */

#include "Spellcast.h"


enum client_events {
  CSOCK_EVENT = 328
};


const int DEFAULT_TIMEOUT = 15;
Client* g_client = NULL;


bool Client::Open(const wxString& name, wxIPV4address addr) {
  wxASSERT(g_client == NULL);
  Client* cl = new Client(name, addr);
  if (cl->is_connected()) {
    g_client = cl;
    return true;
  }
  delete cl;
  return false;
}


Client::Client(const wxString& name, wxIPV4address addr) {
  wxString msg;
  int timeout = DEFAULT_TIMEOUT;

  wxASSERT(g_client == NULL);
  m_state = new GameState();
  m_peer = addr;
  m_sock = new wxSocketClient();
  m_dead = false;
  m_sock->Connect(addr, FALSE);

  wxBeginBusyCursor();
  do {
    msg.Printf(wxT("Connecting to %s... (%d second%s)"), ip_peer_name(addr),
	       timeout, timeout == 1 ? wxT("") : wxT("s"));
    g_window->SetStatus(msg);
    timeout--;
  } while (m_sock->WaitOnConnect(1) == false && timeout > 0);
  wxEndBusyCursor();

  if (m_sock->IsConnected()) {
    m_sock->SetEventHandler(*this, CSOCK_EVENT);
    m_sock->SetNotify(wxSOCKET_INPUT_FLAG | wxSOCKET_LOST_FLAG);
    m_sock->Notify(TRUE);

    send_event(Event::Name(name, GET_MY_GENDER()));
    g_window->SetStatus(wxT("Waiting for confirmation from server..."));
    // We don't report a successful connection until we get the welcome event.

  } else {
    g_window->log(wxT("Couldn't connect to %s: %s."), ip_peer_name(addr),
		  m_sock->Error() ? socket_error_string(m_sock) : wxT("connection timed out"));
  }
}


Client::~Client() {
  SetEvtHandlerEnabled(false);
  if (m_buf.length() > 0) {
    g_window->log(m_buf.as_str());
    m_buf.clear();
  }
  if (m_state) delete m_state;
  if (m_sock) m_sock->Destroy();
  m_sock = NULL;
  m_state = NULL;
  g_client = NULL;
  g_window->set_connected(false);
  g_window->game_over();
}


void Client::send_event(const Event& ev) {
  wxString s;
  MemBuf output = ev.serialize();
  wxASSERT(m_sock != NULL);

  if (m_sock->IsConnected()) {
    // output.append("\n", 1);  no longer using this as a separator
    m_sock->Write(output.data(), output.length());
    if (m_sock->Error()) {
      g_window->log(wxT("Lost connection to server: Write error (%s)"), socket_error_string(m_sock));
      delete this;
    }
    wxLogDebug(wxT("[CLIENT] Sending %s."), event_type_list[ ev.type() ]);

  } else {
    wxLogDebug(wxT("[CLIENT] Can't send %s: we've been disconnected!"),
	       event_type_list[ev.type()]);
  }
}


void Client::send_gestures() {
  int left = m_state->me()->get_gesture(LEFT_HAND, 0);
  int right = m_state->me()->get_gesture(RIGHT_HAND, 0);
  send_event(Event::ClHands(left, right));
}


void Client::_update_state(Event& ev) {
  GameState* old = m_state;
  m_state = new GameState(ev.data(), ev.arg2());
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (m_state->player(i) != NULL) {
      m_state->replace_player(i, old->player(i));
    }
  }
  delete old;
}


void Client::OnSockEvent(wxSocketEvent& event) {
  wxString msg;
  Event ev;

  wxASSERT(m_sock != NULL);
  switch (event.GetSocketEvent()) {
  case wxSOCKET_INPUT:
    if (_eat_some_data() == false) break;
    while (_read_event(ev) == true) {
      wxLogDebug(wxT("[CLIENT] Received %s."), event_type_list[ev.type()]);
      switch (ev.type()) {
      case EVENT_ERROR:
	g_window->log(wxT("Connection failed: %s."), C_STR(ev.data()->as_str()));
	delete this;
	return;

      case EVENT_CHAT:
	g_window->log(wxT("%s says \"%s\""), C_STR(m_state->player(ev.arg1())->name()),
		      C_STR(ev.data()->as_str()));
	break;

      case EVENT_WELCOME: {
	wxIPV4address addr;
	if (!event.GetSocket()->GetPeer(addr)) wxLogFatalError("Can't get peer address!");
	wxString hostname = addr.Hostname();
	if (hostname.length() == 0) hostname = addr.IPAddress();

	m_state = new GameState(ev.data(), ev.arg1());
	msg.Printf(wxT("Connected to %s (%s)."), m_state->server_name().c_str(), hostname.c_str());
	g_window->SetStatus(msg);
	g_window->log(msg);
	g_window->refresh();
      } break;

      case EVENT_JOIN:
	g_window->log(wxT("%s joined the game."), C_STR(ev.data()->as_str()));
	m_state->add_player(ev.arg1(), ev.data()->as_str(), GENDER_MALE);
	g_window->refresh();
	break;

      case EVENT_LEAVE: {
	wxString reason;
	if (ev.data()->length() > 0) {
	  reason.Printf(wxT(" (%s)"), C_STR(ev.data()->as_str()));
	}
	const char* name = C_STR(m_state->player(ev.arg1())->name());
	const char* rc = C_STR(reason);
	g_window->log(wxT("%s left the game%s."), name, rc);

	//	g_window->log(wxT("%s left the game%s."), C_STR(m_state->player(ev.arg1())->name()),
		      //		      C_STR(reason));
	m_state->remove_player(ev.arg1());
	g_window->refresh();   // make sure their name gets erased
      } break;

      case EVENT_NEWTURN: {
	_update_state(ev);
	g_window->turn_started();
	g_window->refresh();
      } break;

      case EVENT_GAMEOVER:
	_update_state(ev);
	g_window->game_over();
	g_window->refresh();
	break;

      case EVENT_MSG:
	g_window->log(wxT("%s"), C_STR(ev.data()->as_str()));
	break;

      case EVENT_SV_HANDS:
	if (ev.data()->length() > 0) {
	  m_state->player(ev.arg1())->set_gesture(LEFT_HAND, ev.arg2());
	  m_state->player(ev.arg1())->set_gesture(RIGHT_HAND, ev.arg3());
	} else {
	  m_state->player(ev.arg1())->add_gestures(ev);
	}
	g_window->refresh();
	break;

      case EVENT_WAITING:
	m_state->set_done_players(ev.arg1());
	g_window->refresh();
	break;

      case EVENT_QUESTION: {
	Question q(ev.data()->as_str());
	g_window->show_question(Question(ev.data()->as_str()));
	g_window->turn_started();
      } break;

      default:
	FAIL();
      }
    }
    break;

  case wxSOCKET_LOST:
    msg.Printf(wxT("Lost connection to server: %s"),
	       m_sock->Error() ? socket_error_string(m_sock) : wxT("Server quit."));
    g_window->log(msg);
    delete this;
    return;

  default:
    wxLogDebug(wxT("WHAT THE FUCK?"));
    FAIL();
  }

  if (m_dead) {
    msg.Printf(wxT("Lost connection to server: Read error (%s)"), socket_error_string(m_sock));
    g_window->log(msg);
    delete this;
  }
}


static const int READBUF_LEN = 1024;

bool Client::_eat_some_data() {
  wxUint8 readbuf[READBUF_LEN];

  wxASSERT(m_sock != NULL);
  m_sock->Read(readbuf, READBUF_LEN);
  m_dead = m_sock->Error();
  m_buf.append(readbuf, m_sock->LastCount());

  if (m_buf.length() > READBUF_LEN * 2) {
    g_window->log(wxT("Lost connection to server: Read error (Data flood)"));
    delete this;
    return false;
  }
  return true;
}


bool Client::_read_event(Event& ev) {
  ev = Event(&m_buf);
  return ev.is_ok();
}


BEGIN_EVENT_TABLE(Client, wxEvtHandler)
  EVT_SOCKET(CSOCK_EVENT, Client::OnSockEvent)
END_EVENT_TABLE()
