/* $Id: Util.cpp,v 1.10 2008/05/04 09:19:51 dpt Exp $
 *
 * Util.cpp: Miscellaneous useful functions.
 */

#include "Util.h"


static void _serialize_string(const wxString& input, MemBuf* output) {
  wxUint32 len = wxUINT32_SWAP_ON_LE(input.length());
  // size_t old = output->length();
  output->append((const wxUint8*) &len, 4);
  output->append(input);
  // fprintf(stderr, "serialize: \"%s\", %d chars. (%d + %d = %d)\n", C_STR(input), input.length(), old, input.length() + 4, output->length());
}


MemBuf serialize(const char* fmt, ...) {
  MemBuf output;

  wxASSERT(fmt != NULL);

  va_list ap;
  va_start(ap, fmt);
  for (int i = 0 ; fmt[i] != '\0' ; i++) {
    switch (fmt[i]) {
    case 'b': case 'B': {
      wxInt8 data = va_arg(ap, int);
      output.append((const wxUint8*) &data, 1);
    } break;
    case 's': {
      wxInt16 data = va_arg(ap, int);
      data = wxINT16_SWAP_ON_LE(data);
      output.append((const wxUint8*) &data, 2);
    } break;
    case 'S': { 
      wxUint16 data = va_arg(ap, int);
      data = wxUINT16_SWAP_ON_LE(data);
      output.append((const wxUint8*) &data, 2);
    } break;
    case 'i': {
      wxInt32 data = va_arg(ap, int);
      data = wxINT32_SWAP_ON_LE(data);
      output.append((const wxUint8*) &data, 4);
    } break;
    case 'I': {
      wxUint32 data = va_arg(ap, int);
      data = wxUINT32_SWAP_ON_LE(data);
      output.append((const wxUint8*) &data, 4);
    } break;
    case 'M': {
      MemBuf* memp = va_arg(ap, MemBuf*);
      wxUint32 len = wxUINT32_SWAP_ON_LE(memp->length());
      output.append((const wxUint8*) &len, 4);
      output.append(memp->data(), memp->length());
    } break;
    case 'T': {
      wxString* input = va_arg(ap, wxString*);
      _serialize_string(*input, &output);
    } break;
    default:
      wxLogFatalError(wxT("Bogus character in serialize format string: %c"), fmt[i]);
    }
  }
  va_end(ap);
  return output;
}


void serialize_string_array(wxString array[], wxUint32 howmany, MemBuf* output) {
  wxASSERT(array != NULL);
  for (wxUint32 i = 0 ; i < howmany ; i++) {
    _serialize_string(array[i], output);
  }
}


static wxUint32 _read_string(const wxUint8* input, wxUint32 input_len, wxString* output) {
  wxUint32 len = wxINT32_SWAP_ON_LE(*((wxUint32*) input));
  if (len > input_len - 4) return 0;
  *output = wxString((const char*) input + 4, wxConvUTF8, len);
  // fprintf(stderr, "unserialize: \"%s\", %d chars. (len %d)\n", C_STR(*output), output->length(), len);
  return len + 4;
}


wxUint32 unserialize_string_array(const wxUint8* input, wxUint32 input_len,
				  wxString array[], wxUint32 howmany) {
  wxUint32 index = 0;
  for (wxUint32 i = 0 ; i < howmany ; i++) {
    wxUint32 len = _read_string(input + index, input_len - index, &array[i]);
    if (len == 0) return 0;
    index += len;
  }
  return index;
}


wxUint32 unserialize(MemBuf* input, const char* fmt, ...) {
  wxUint32 index = 0;
  va_list ap;

  wxASSERT(fmt != NULL);

  va_start(ap, fmt);
  for (size_t x = 0 ; fmt[x] != '\0' ; x++) {
    if (index >= input->length()) {
      return 0;
    }
    switch (fmt[x]) {
    case 'b': {
      wxInt8* data = va_arg(ap, wxInt8*);
      *data = (*input)[index++];
    } break;
    case 'B': {
      wxUint8* data = va_arg(ap, wxUint8*);
      *data = (*input)[index++];
    } break;
    case 's': {
      wxInt16* data = va_arg(ap, wxInt16*);
      memmove(data, input->data() + index, 2);
      *data = wxINT16_SWAP_ON_LE(*data);
      index += 2;
    } break;
    case 'S': {
      wxUint16* data = va_arg(ap, wxUint16*);
      memmove(data, input->data() + index, 2);
      *data = wxUINT16_SWAP_ON_LE(*data);
      index += 2;
    } break;
    case 'i': {
      wxInt32* data = va_arg(ap, wxInt32*);
      memmove(data, input->data() + index, 4);
      *data = wxINT32_SWAP_ON_LE(*data);
      index += 4;
    } break;
    case 'I': {
      wxUint32* data = va_arg(ap, wxUint32*);
      memmove(data, input->data() + index, 4);
      *data = wxUINT32_SWAP_ON_LE(*data);
      index += 4;
    } break;
    case 'M': {
      MemBuf* memp = va_arg(ap, MemBuf*);
      wxUint32 len = 0;
      memmove((void*) &len, input->data() + index, 4);
      len = wxUINT32_SWAP_ON_LE(len);
      memp->clear();
      memp->append(input->data() + index + 4, len);
      index += len + 4;
    } break;
    case 'T': {
      wxUint32 i;
      wxString s;
      wxString* data = va_arg(ap, wxString*);
      if ((i = _read_string(input->data() + index, input->length() - index, &s)) == 0) {
	return 0;
      }
      *data = s;
      index += i;
    } break;

    default:
      wxLogFatalError(wxT("Bogus character in unserialize format string: %c"), fmt[x]);
    }
  }
  va_end(ap);
  input->trim_start(index);
  return index;
}



// A simple self-test...

class Testdata {
 public:
  Testdata() {
    a = -3214243; b = 2147483612; c = -1;
    d = 0; e = 3; f = 1829966282; f *= 2;
    g = wxT(""); h = wxT("\\\"Monkey monkey monkey!\\\"");
    i = wxT("I\\n\t\\\"have\\\" a \\ferret\\ up my \\\\nose.\\\"");
    j = -1; k = 127; l = 47;
  }
  Testdata(MemBuf* input) {
    if (unserialize(input, "iiiIIITTTbbb", &a, &b, &c, &d, &e, &f, &g, &h, &i, &j, &k, &l) == 0) {
      wxLogFatalError(wxT("test_serialize failed!"));
      wxExit();
    }
  }
  bool operator==(const Testdata& other) const {
    return (a == other.a && b == other.b && c == other.c && d == other.d && e == other.e &&
	    f == other.f && g == other.g && h == other.h && i == other.i && j == other.j &&
	    k == other.k && l == other.l);
  }
  bool operator!=(const Testdata& other) const {
    return !(*this == other);
  }
  MemBuf serialize() const {
    return ::serialize("iiiIIITTTbbb", a, b, c, d, e, f, &g, &h, &i, j, k, l);
  }

 private:
  wxInt32  a, b, c;
  wxUint32 d, e, f;
  wxString g, h, i;
  wxInt8   j, k, l;
};


bool test_serialize() {
  MemBuf mb;
  Testdata test1;       mb = test1.serialize();
  Testdata test2(&mb);  mb = test2.serialize();
  Testdata test3(&mb);  mb = test3.serialize();
  Testdata test4(&mb);

  if (test1 != test2 || test1 != test3 || test1 != test4) {
    wxLogFatalError(wxT("multiple serializing test failed!"));
    return false;
  }
  return true;
}


const wxChar* socket_error_string(wxSocketBase* sock) {
  if (sock->Error()) {
    switch (sock->LastError()) {
    case wxSOCKET_NOERROR:    return wxT("No error");
    case wxSOCKET_INVOP:      return wxT("Invalid operation");
    case wxSOCKET_IOERR:      return wxT("I/O error");
    case wxSOCKET_INVADDR:    return wxT("Invalid address");
    case wxSOCKET_INVSOCK:    return wxT("Uninitialized socket");
    case wxSOCKET_NOHOST:     return wxT("No such host");
    case wxSOCKET_INVPORT:    return wxT("Invalid port");
    case wxSOCKET_WOULDBLOCK: return wxT("Operation would block");
    case wxSOCKET_TIMEDOUT:   return wxT("Operation timed out");
    case wxSOCKET_MEMERR:     return wxT("Memory exhausted");
    case wxSOCKET_DUMMY:      return wxT("I LIKE PIE");
    }
  }
  return wxT("No error");
}


const wxChar* ip_peer_name(wxIPV4address addr) {
  return addr.Hostname().length() == 0 ? addr.IPAddress().c_str() : addr.Hostname().c_str();
}


int StringFind(const wxString& str, char ch) {
  for (size_t i = 0 ; i < str.Len() ; i++) {
    if (str[i] == ch) {
      return i;
    }
  }
  return -1;
}


void StringRemove(wxString& str, size_t pos, size_t len) {
  wxASSERT(len > 0);
  wxASSERT(pos < str.Len());

  wxString out;
  for (size_t i = 0 ; i < str.Len() ; i++) {
    if (i < pos || i >= pos + len) {
      out += str[i];
    }
  }
  str = out;
}


wxString StringReverse(const wxString& str) {
  wxString s;

  for (size_t i = 0 ; i < str.length() ; i++) {
    s.Prepend(str[i]);
  }
  return s;
}


// just for debugging
wxString escaped_string(const wxString& str) {
  wxString zot;
  for (size_t i = 0 ; i < str.Len() ; i++) {
    if (str[i] == '\n') {
      zot += '&';
    } else if (isspace(str[i])) {
      zot += '_';
    } else if (isprint(str[i])) {
      zot += str[i];
    } else {
      zot += '^';
    }
  }
  return zot;
}


wxSize find_max_widget_size(wxSize a, wxSize b) {
  if (a.GetWidth() * a.GetHeight() > b.GetWidth() * b.GetHeight()) {
    return a;
  }
  return b;
}


void fail(const wxChar* fmt, ...) {
  wxString str;
  va_list ap;

  va_start(ap, fmt);
  str.PrintfV(fmt, ap);
  str += '\n';
  fprintf(stderr, str.char_str());
  wxLogError(str);
  va_end(ap);
  // abort();  this doesn't invoke the debugger under cygwin
  int i = *((int*)NULL);    // betcha this does! :-D
  printf("%d", i);   // to suppress unused variable optimization
}
