// Server.h: The game server itself, arbiter of all rules.

#include "Spellcast.h"


const int LOGIN_WAIT_SECONDS = 10;
const int N_BACKUP_NAMES = 16;
Server* g_server = NULL;

// Ran out of colors after a while, so I just raided X11/rgb.txt. :-)
static const char* backup_names[ N_BACKUP_NAMES ] = {
  "White Mage", "Black Mage", "Red Mage", "Green Mage", "Blue Mage",
  "Yellow Mage", "Grey Mage", "Purple Mage", "Fuschia Mage", "Taupe Mage",
  "Burnt Sienna Mage", "DarkOrchid4 Mage", "Violet Mage", "Goldenrod Mage",
  "Grey99 Mage", "LightSteelBlue Mage"
};

enum server_events {
  CLIENT_EVENT = 326,
  SERVER_EVENT
};


enum connection_states {
  CONNECTION_REJECTED,
  CONNECTION_UP,
  CONNECTION_LOGGED_IN,
  NUMBER_OF_CONNECTION_STATES
};


class Connection {
 public:
  Connection(wxSocketBase* sock) {
    wxASSERT(sock != NULL);
    m_sock = sock;
    m_timer = new wxStopWatch;
    m_timer->Start();
    m_state = CONNECTION_UP;
  }
  ~Connection() {
    m_sock->Destroy();
    delete m_timer;
  }
  void sanity_check() const {
    if (m_state < 0 || m_state >= NUMBER_OF_CONNECTION_STATES) { abort(); }
  }

  inline wxSocketBase* sock() const { return m_sock; }
  inline void login() { m_state = CONNECTION_LOGGED_IN; }
  inline void reject() { m_state = CONNECTION_REJECTED; }
  inline bool is_rejected() const { return m_state == CONNECTION_REJECTED; }
  inline bool is_logged_in() const { return m_state == CONNECTION_LOGGED_IN; }
  inline int seconds_up() const { return m_timer->Time() / 1000; }
  MemBuf buf;

 private:
  wxSocketBase* m_sock;
  wxStopWatch*  m_timer;
  int m_state;
};


bool Server::Open(const wxString& sv_name, int max_players, wxUint16 port) {
  wxASSERT(g_server == NULL);
  g_server = new Server(sv_name, max_players, port);
  if (g_server->m_sock == NULL) {
    delete g_server;
    g_server = NULL;
    return false;
  }
  return true;
}


Server::Server(const wxString& sv_name, int max_players, wxUint16 port) {
  wxIPV4address addr;
  addr.AnyAddress();
  addr.Service(port);

  wxASSERT(g_server == NULL);
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    m_clients[i] = NULL;
  }

  m_rules = NULL;
  m_suppliants.Empty();

  m_sock = new wxSocketServer(addr);
  if (m_sock->Ok() == false) {
    wxLogError(wxT("Failed to open listening socket on port %d: %s"), port, socket_error_string(m_sock));
    m_sock->Destroy();
    m_sock = NULL;
  } else {
    m_sock->SetEventHandler(*this, SERVER_EVENT);
    m_sock->SetNotify(wxSOCKET_CONNECTION_FLAG | wxSOCKET_LOST_FLAG);
    m_sock->Notify(TRUE);

    m_rules = new Rules();
    m_rules->set_max_players(max_players);
    m_rules->set_game_state(GAME_NOT_STARTED);
    m_rules->set_server_name(sv_name);
    m_rules->set_server_port(port);
  }
}


Server::~Server() {
  wxASSERT(g_server != NULL);
  SetEvtHandlerEnabled(false);
  if (m_rules) {
    delete m_rules;
    m_rules = NULL;
  }
  if (m_sock) {
    g_window->log(wxT("Shutting down server."));
    m_sock->Destroy();
    m_sock = NULL;
  }
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (m_clients[i] != NULL) {
      delete m_clients[i];
    }
  }
  for (size_t i = 0 ; i < m_suppliants.GetCount() ; i++) {
    delete m_suppliants[i];
  }

  g_server = NULL;
  g_window->set_connected(false);
}


bool Server::is_full() const {
  return m_rules->player_count() == m_rules->max_players();
}


void Server::send_event(int client_no, const Event& ev) {
  if (client_no < 0 || (unsigned) client_no >= MAX_PLAYERS || m_clients[ client_no ] == NULL) {
    g_window->log(wxT("Sending server data() to nonexistent client %d!"), client_no);
    delete this;
    return;
  }

  _send_data(m_clients[ client_no ], ev.serialize());
  wxLogDebug(wxT("[SERVER] Sent %s to client %d."), event_type_list[ ev.type() ], client_no);
}


void Server::broadcast(const Event& ev) {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (m_clients[i] != NULL) {
      wxASSERT(m_clients[i]->is_logged_in());
      _send_data(m_clients[i], ev.serialize());
    }
  }
  wxLogDebug(wxT("[SERVER] Broadcasting %s to all clients."), event_type_list[ ev.type() ]);
}


wxUint16 Server::port() const {
  wxIPV4address addr;
  wxASSERT(m_sock->GetPeer(addr) == true);
  return addr.Service();
}


void Server::_sanity_check() const {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (m_clients[i] != NULL) {
      m_clients[i]->sanity_check();
    }
  }

  for (size_t i = 0 ; i < m_suppliants.GetCount() ; i++) {
    m_suppliants[i]->sanity_check();
  }
}


void Server::OnClientEvent(wxSocketEvent& event) {
  Connection* conn = _socket_to_conn(event.GetSocket());
  int client_no = _conn_to_client(conn);
  wxString err;
  Event ev;

  switch (event.GetSocketEvent()) {
  case wxSOCKET_INPUT:
    if (_eat_some_data(conn) == false) { break; }
    while (_read_event(conn, ev) == true) {
      wxLogDebug(wxT("[SERVER] Received %s from client %d."),
		 event_type_list[ ev.type() ], client_no);
      switch (ev.type()) {
      case EVENT_CHAT:
	if (client_no < 0) {
	  _remove_connection(conn);
	}
	broadcast(Event::Chat(ev.data()->as_str(), client_no));
	break;

      case EVENT_NAME:
	if (client_no >= 0) {
	  goto BOGUS;
	}
	if (conn->is_rejected() == false) {
	  wxIPV4address addr;
	  if (event.GetSocket()->GetPeer(addr) == false) {
	    _send_data(conn, Event::Error(wxT("Couldn't get your address!")).serialize());
	    conn->reject();
	  }
	  if (is_full()) {
	    wxASSERT(m_rules->player_count() == m_rules->max_players());
	    _send_data(conn, Event::Error(wxT("The server is full")).serialize());
	    conn->reject();

	  } else if (m_rules->game_state() != GAME_NOT_STARTED) {
	    _send_data(conn, Event::Error(wxT("Game is already in progress")).serialize());
	    conn->reject();

	  } else {
	    client_no = _next_client_slot();
	    wxASSERT(client_no >= 0);
	    Player* newplayer = new Player(ev);
	    if (m_rules->name_is_taken(ev.data()->as_str())) {
	      int i;
	      for (i = 0 ; i < N_BACKUP_NAMES ; i++) {
		if (m_rules->name_is_taken(backup_names[i]) == false) {
		  delete newplayer;
		  newplayer = new Player(backup_names[i], ev.arg1());
		  break;
		}
	      }
	      REQUIRE(i < N_BACKUP_NAMES);
	    }
	    broadcast(Event::Join(client_no, newplayer->name()));
	    m_suppliants.Remove(conn);
	    m_clients[ client_no ] = conn;
	    m_clients[ client_no ]->login();
	    m_rules->add_player(client_no, newplayer);
	    _send_data(conn, Event::Welcome(client_no, *m_rules).serialize());
	    if (newplayer->name() != ev.data()->as_str()) {
	      _send_data(conn, Event::Msg(wxString::Format(wxT("The name \"%s\" is already taken. You have been assigned the name \"%s\" instead."), C_STR(ev.data()->as_str()), C_STR(newplayer->name()))).serialize());
	    }
	  }
	}
	break;

      case EVENT_CL_HANDS:
	if (m_rules->game_state() == GAME_IN_PROGRESS && m_rules->phase() == GESTURES_PHASE) {
	  m_rules->add_gestures(client_no, ev.arg2(), ev.arg3());
	  _do_turn();
	} else {
	  goto BOGUS;
	}
	break;

      case EVENT_ANSWER:
	if (m_rules->game_state() == GAME_IN_PROGRESS && m_rules->phase() == ANSWERS_PHASE) {
	  m_rules->add_answer(client_no, ev.data()->as_str(), ev.arg1());
	  _do_turn();
	} else {
	  goto BOGUS;
	}
	break;

      default:
      BOGUS:
	err.Printf(wxT("Got bogus packet type: %d"), ev.type());
	_remove_connection(conn, err);
      }
    }
    break;

  case wxSOCKET_LOST:
    err.Printf(wxT("Lost connection (%s)"), m_sock->Error() ? socket_error_string(m_sock) : wxT("Client disconnected"));
    _remove_connection(conn, err);
    break;

  default:
    FAIL();
  }
  _clean_suppliant_list();   // remove any rejected sockets now
  wxLogDebug(wxT("[SERVER] ClientEvent: Got %d clients and %d suppliants."),
	      m_rules->player_count(), m_suppliants.GetCount());
}


void Server::OnServerEvent(wxSocketEvent& event) {
  wxSocketBase* sock;
  Connection* conn;

  _sanity_check();
  switch (event.GetSocketEvent()) {
  case wxSOCKET_CONNECTION:
    sock = m_sock->Accept(false);
    if (sock == NULL) {
      if (m_clients[0] == NULL) {
	g_window->log(wxT("Couldn't accept local client: %s"), socket_error_string(m_sock));
	delete this;
	return;
      }
      wxString err;
      err.Printf(wxT("Couldn't accept connection: %s"), socket_error_string(m_sock));
      _send_data(m_clients[0], Event::Error(err).serialize());
      break;
    }

    sock->SetEventHandler(*this, CLIENT_EVENT);
    sock->SetNotify(wxSOCKET_INPUT_FLAG | wxSOCKET_LOST_FLAG);
    sock->Notify(true);

    conn = new Connection(sock);
    _sanity_check();
    m_suppliants.Add(conn);
    if (is_full()) {
      _send_data(conn, Event::Error(wxT("The server is full")).serialize());
      conn->reject();
    }
    _sanity_check();
    break;

  case wxSOCKET_LOST:
    g_window->log(wxT("Server socket unexpectedly died: %s"), socket_error_string(m_sock));
    delete this;
    return;

  default:
    FAIL();
  }

  _clean_suppliant_list();
  _sanity_check();
  wxLogDebug(wxT("[SERVER] ServerEvent: Got %d clients and %d suppliants."),
	      m_rules->player_count(), m_suppliants.GetCount());
}


void Server::_send_data(Connection* conn, const MemBuf& data) {
  wxASSERT(conn != NULL);
  conn->sock()->Write(*data, data.length());
  if (conn->sock()->Error()) {
    _remove_connection(conn, wxString::Format(wxT("Write error (%s)"), socket_error_string(m_sock)));
  }
}


void Server::_clean_suppliant_list() {
  Connection* conn;

  for (size_t i = 0 ; i < m_suppliants.GetCount() ; i++) {
    conn = m_suppliants[i];
    if (conn->is_rejected() || conn->seconds_up() > LOGIN_WAIT_SECONDS) {
      _remove_connection(conn);

    } else if (is_full()) {
      _send_data(conn, Event::Error(wxT("The server is full")).serialize());
      conn->reject();
    }
  }
}


static const int READBUF_LEN = 1024;

bool Server::_eat_some_data(Connection* conn) {
  wxUint8 readbuf[ READBUF_LEN ];

  conn->sock()->Read(readbuf, READBUF_LEN);
  if (conn->sock()->Error()) {
    _remove_connection(conn, wxString::Format(wxT("Read error (%s)"), socket_error_string(m_sock)));
    return false;
  }

  conn->buf.append(readbuf, conn->sock()->LastCount());
  if (conn->buf.length() > READBUF_LEN * 2) {
    _remove_connection(conn, wxT("Data flood"));
    return false;
  }
  return true;
}


bool Server::_read_event(Connection* conn, Event& ev) {
  ev = Event(&conn->buf);
  return ev.is_ok();
}


Connection* Server::_socket_to_conn(wxSocketBase* sock) {
  wxASSERT(sock != NULL);

  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (m_clients[i] != NULL && sock == m_clients[i]->sock()) {
      return m_clients[i];
    }
  }

  for (size_t i = 0 ; i < m_suppliants.GetCount() ; i++) {
    if (sock == m_suppliants[i]->sock()) {
      return m_suppliants[i];
    }
  }

  g_window->log(wxT("[SERVER] No matching socket! (can't happen)"));
  delete this;
  return NULL;
}


int Server::_conn_to_client(Connection* conn) {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (conn == m_clients[i]) {
      return i;
    }
  }
  return -1;
}


int Server::_next_client_slot() {
  return _conn_to_client(NULL);
}


void Server::_remove_connection(Connection* conn, const wxString& reason) {
  int i = _conn_to_client(conn);
  wxASSERT(conn != NULL);

  if (i < 0) {
    m_suppliants.Remove(conn);
    wxLogDebug(wxT("[SERVER] Destroyed a suppliant's connection!"));

  } else {
    m_clients[i] = NULL;
    broadcast(Event::Leave(i, reason));
    m_rules->remove_player(i);
  }

  delete conn;
}


void Server::_do_turn() {
  while (m_rules->phase_is_done() && m_rules->game_state() != GAME_OVER) {
    switch (m_rules->phase()) {
    case GESTURES_PHASE:
      m_rules->start_questions();
      break;

    case ANSWERS_PHASE:
      m_rules->run_turn_events();
      if (m_rules->game_state() != GAME_OVER) {
	m_rules->next_turn();
      }
      break;

    default: FAIL();
    }
  }
}




BEGIN_EVENT_TABLE(Server, wxEvtHandler)
  EVT_SOCKET(CLIENT_EVENT, Server::OnClientEvent)
  EVT_SOCKET(SERVER_EVENT, Server::OnServerEvent)
END_EVENT_TABLE()
