/* $Id: Rules.cpp,v 1.11 2008/05/04 09:19:51 dpt Exp $
 *
 * Rules.cpp: a class that encapsulates the game logic.
 */

#include "Messages.h"
#include "Spellcast.h"


Rules::Rules() {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    m_questions[i] = NULL;
  }
  m_spells.DeleteContents(true);
}


Rules::Rules(const Rules& other) : GameState(other) {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    m_questions[i] = other.m_questions[i];
  }

  m_spells.DeleteContents(true);
  for (DoneSpellList::Node* node = other.m_spells.GetFirst() ; node ; node = node->GetNext()) {
    DoneSpell* sp = new DoneSpell;
    *sp = *node->GetData();
    m_spells.Append(sp);
  }
}


Rules& Rules::operator=(const Rules& other) {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (m_questions[i] != NULL) {
      delete m_questions[i];
    }
    m_questions[i] = other.m_questions[i];
  }

  m_spells.Clear();
  for (DoneSpellList::Node* node = other.m_spells.GetFirst() ; node ; node = node->GetNext()) {
    DoneSpell* sp = new DoneSpell;
    *sp = *node->GetData();
    m_spells.Append(sp);
  }
  return *this;
}


Rules::~Rules() {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (m_questions[i] != NULL) {
      delete m_questions[i];
    }
  }
}


void Rules::_send_turn() const {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL) {
      g_server->send_event(i, Event::NewTurn(i, *this));
    }
  }
}


#define SEND_NORMAL() \
  g_server->send_event(i, Event::SvHands(x, player(x)->get_gesture(LEFT_HAND, 0), \
                                         player(x)->get_gesture(RIGHT_HAND, 0), replace))
#define SEND_UNKNOWN() \
  g_server->send_event(i, Event::SvHands(x, GESTURE_UNKNOWN, GESTURE_UNKNOWN, replace))


void Rules::_send_gestures(bool replace, int id) const {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) == NULL) { continue; }
    for (unsigned int x = 0 ; x < MAX_PLAYERS ; x++) {
      if (player(x) == NULL || (id >= 0 && (int) x != id)) { continue; }

      // In a time-stopped turn, everyone else just sees that person get
      // blank spaces in their gesture list.
      if (m_turn_type == TURN_TIMESTOP) {
	if (player(x)->duration(D_TIMESTOP)) {
	  if (i == x) {
	    SEND_NORMAL();
	  } else {
	    SEND_UNKNOWN();
	  }
	}

      // In a hasted turn, everyone sees the speedy person's extra gestures.
      } else if (m_turn_type == TURN_HASTED) {
	if (player(x)->duration(D_HASTE)) {
	  if ((player(i)->duration(D_BLINDNESS) || player(x)->duration(D_INVISIBILITY)) && x != i) {
	    SEND_UNKNOWN();    // This player is invisible, or 'p' is blind.
	  } else {
	    SEND_NORMAL();
	  }
	}

      // Normal turn:
      } else if ((player(i)->duration(D_BLINDNESS) || player(x)->duration(D_INVISIBILITY)) && x != i) {
	SEND_UNKNOWN();      // This player is invisible, or 'p' is blind.
      } else {
	SEND_NORMAL();
      }
    }
  }
}


void Rules::next_turn() {
  wxASSERT(g_server != NULL);

  set_phase(GESTURES_PHASE);
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    Player* p = player(i);
    if (p != NULL && p->active()) {
      p->scroll_up();
      p->set_done(false);
    }
  }

  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) == NULL) { continue; }
    for (unsigned int x = 0 ; x < MAX_PLAYERS ; x++) {
      if (player(x)) {
	if (player(x)->active()) {
	  if (i == x) {
	    g_server->send_event(i, Event::SvHands(x, GESTURE_NOTHING, GESTURE_NOTHING, false));
	  } else {
	    g_server->send_event(i, Event::SvHands(x, GESTURE_UNKNOWN, GESTURE_UNKNOWN, false));
	  }
	} else if (!player(x)->is_alive()) {
	  g_server->send_event(i, Event::SvHands(x, GESTURE_NOTHING, GESTURE_NOTHING, false));
	}
      }
    }
  }
  _send_turn();
}


void Rules::start_game() {
  wxASSERT(g_server != NULL);
  wxASSERT(m_game_state == GAME_NOT_STARTED);
  wxASSERT(m_turn == 0);

  m_game_state = GAME_IN_PROGRESS;
  m_turn_type = TURN_NORMAL;

  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL) {
      player(i)->set_active(true);
      player(i)->use_default_gestures();
      g_server->send_event(i, msg_s_game_start());
      for (unsigned int j = 0 ; j < MAX_PLAYERS ; j++) {
	if (player(j) != NULL && j != i) {
	  g_server->send_event(i, msg_o_game_start(player(j)));
	}
      }
    }
  }

  next_turn();
}


void Rules::add_gestures(int id, wxInt8 left, wxInt8 right) {
  wxASSERT(player(id) != NULL);

 if (id >= 0 && id < (int) MAX_PLAYERS && left >= 0 && right >= 0 &&
      left < NUMBER_OF_GESTURES && right < NUMBER_OF_GESTURES && player(id)->is_alive()) {
   player(id)->set_done(true);
   player(id)->set_gesture(LEFT_HAND, left);
   player(id)->set_gesture(RIGHT_HAND, right);
   _send_waiting_events();

  } else {
    g_window->log(wxT("WARNING: Bad gesture packet from client %d: [%d, %d]"), id, left, right);
  }
}


void Rules::remove_player(wxUint8 id) {
  // Delete anything this player did, or was in the middle of doing,
  // this round.
  GameState::remove_player(id);
  if (m_game_state == GAME_IN_PROGRESS) {
    if (m_questions[id]) m_questions[id]->Clear();
    for (DoneSpellList::Node* node = m_spells.GetFirst() ; node ; node = node->GetNext()) {
      DoneSpell* sp = node->GetData();
      if (sp->caster == id) m_spells.Erase(node);
    }
    is_game_over();
  }
}


void Rules::_add_elemental_question(int id, wxUint8 sp) {
  wxASSERT(m_spells.Item(sp)->GetData()->id == SPELL_SUMMON_ELEMENTAL);
  static const wxString choices[2] = { wxT("Fire"), wxT("Ice") };
  Question* q = new Question(wxT("Which type of elemental do you want to summon?"), 2, choices);
  q->set_type(QUESTION_ELEMENTAL);
  q->set_spell(sp);
  _add_question(id, q);
}


void Rules::_add_new_monster_question(int id, wxUint8 sp) {
  const char* name = NULL;
  switch (m_spells.Item(sp)->GetData()->id) {
  case SPELL_SUMMON_GOBLIN:  name = "Goblin"; break;
  case SPELL_SUMMON_OGRE:    name = "Ogre";   break;
  case SPELL_SUMMON_TROLL:   name = "Troll";  break;
  case SPELL_SUMMON_GIANT:   name = "Giant";  break;
  default: FAIL();
  }

  wxString foo;
  foo.Printf(wxT("Whom do you want your newly-summoned %s to attack?"), name);
  Question* q = new Question(foo);
  q->set_type(QUESTION_NEW_MONSTER);
  q->set_spell(sp);

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i) != NULL && being(i)->is_alive() && (int) i != id) {
      q->add_answer(being(i)->name());
    }
  }
  q->add_answer(wxT("(nobody)"));
  _add_question(id, q);
}


void Rules::_add_monster_question(int id, wxUint8 monster) {
  wxString foo = wxString::Format(wxT("Whom do you want %s to attack?"),
				  C_STR(creature(monster)->name()));
  Question* q = new Question(foo);
  q->set_type(QUESTION_MONSTER);
  q->set_monster(monster);

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if (being(i) != NULL && (int) i != id && i != monster + MAX_PLAYERS && being(i)->is_alive()) {
      q->add_answer(being(i)->name());
    }
  }
  q->add_answer(wxT("(nobody)"));
  _add_question(id, q);
}


void Rules::_add_hand_question(int id, wxUint8 sp) {
  static const wxString choices[2] = { wxT("Left"), wxT("Right") };
  wxString foo, action;
  DoneSpell* spell = m_spells.Item(sp)->GetData();

  switch (spell->id) {
  case SPELL_PARALYSIS:     action = wxT("paralyze");  break;
  case SPELL_CHARM_PERSON:  action = wxT("charm");     break;
  default:  FAIL();
  }
  foo.Printf(wxT("Which of %s's hands do you want to %s?"),
	     C_STR(being(spell->target)->name()), C_STR(action));
  Question* q = new Question(foo, 2, choices);
  q->set_type(QUESTION_HAND);
  q->set_spell(sp);
  _add_question(id, q);
}


void Rules::_add_charm_question(int id, wxUint8 sp) {
  static wxString choices[ NUMBER_OF_PLAYER_GESTURES - 1 ];
  wxString foo;
  DoneSpell* spell = m_spells.Item(sp)->GetData();

  for (unsigned int i = 0 ; i < NUMBER_OF_PLAYER_GESTURES - 1 ; i++) {
    choices[i] = gesture_name(i + 1);
    choices[i].SetChar(0, toupper(choices[i].GetChar(0)));
  }

  foo.Printf(wxT("What gesture do you want %s' %s to make?"),
	     C_STR(being(spell->target)->name()), which_hand(spell->mind_hand));
  Question* q = new Question(foo, NUMBER_OF_PLAYER_GESTURES - 1, choices);
  q->set_type(QUESTION_CHARM);
  q->set_spell(sp);
  _add_question(id, q);
}


void Rules::_add_target_question(int id, int hand, wxUint8 sp) {
  wxString foo;
  DoneSpell* spell = m_spells.Item(sp)->GetData();

  if (spell->id == SPELL_STAB) {
    foo.Printf(wxT("Whom do you want to stab at with %s?"), your_hand(hand));
  } else {
    foo.Printf(wxT("Whom do you want to cast %s at with %s?"),
		spell_list[ spell->id ].name, your_hand(hand));
  }
  Question* q = new Question(foo);
  q->set_type(QUESTION_TARGET);
  q->set_spell(sp);

  if (spell_list[ spell->id ].flags & SF_SELF) {
    q->add_answer(player(id)->name());
  }

  for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
    if ((int) i == id) { continue; }
    if (being(i) != NULL &&
	((spell->id == SPELL_RAISE_DEAD && being(i)->raiseable()) ||
	 being(i)->is_alive())) {
      q->add_answer(being(i)->name());
    }
  }

  if (!(spell_list[ spell->id ].flags & SF_SELF)) {
    q->add_answer(player(id)->name());
  }

  q->add_answer(wxT("(nobody)"));
  _add_question(id, q);
}


void Rules::_add_spell_question(int id, PossibleSpellList* list, int hand, bool or_both) {
  DoneSpell* sp = new DoneSpell;
  sp->id = -1;
  sp->target = sp->detail = -2;
  sp->caster = id;
  sp->hand = hand;
  sp->mind_hand = NEITHER_HAND;
  sp->permanent = sp->delayed = sp->n_questions = 0;
  m_spells.Append(sp);

  wxString foo = wxString::Format(wxT("Which spell do you want to cast with %s?"),
				  your_hand(hand));
  Question* q = new Question(foo);
  q->set_type(QUESTION_SPELL);
  q->set_spell(m_spells.GetCount() - 1);

  for (PossibleSpellList::Node* node = list->GetLast() ; node ; node = node->GetPrevious()) {
    if (node->GetData()->hand == hand || (or_both && node->GetData()->hand == BOTH_HANDS)) {
      q->add_answer(spell_list[ node->GetData()->id ].name);
    }
  }
  player(id)->add_spell_question();
  _add_question(id, q, true);
}


void Rules::_add_question(int id, Question* q, bool important) {
  if (m_questions[id] == NULL) {
    m_questions[id] = new QueryList();
    m_questions[id]->DeleteContents(TRUE);
    player(id)->set_done(false);
  }
  if (important) {
    m_questions[id]->Insert(q);
  } else {
    m_questions[id]->Append(q);
  }
  if (q->spell() >= 0) {
    m_spells.Item(q->spell())->GetData()->n_questions++;
  }
}


void Rules::_add_spell(wxUint8 caster, wxUint8 hand, wxUint8 sp_id) {
  wxASSERT(caster < MAX_PLAYERS);
  wxASSERT(sp_id < NUMBER_OF_SPELLS);

  DoneSpell* sp = new DoneSpell;
  sp->id = sp_id;
  sp->hand = hand;
  sp->target = -2;   // -1 means nobody; -2 is invalid.
  sp->caster = caster;
  sp->mind_hand = 0;
  sp->detail = -2;
  sp->n_questions = 0;
  sp->permanent = 0;
  sp->delayed = 0;
  m_spells.Append(sp);
}


bool Rules::_ask_questions(wxUint8 sp) {
  bool asking = false;

  DoneSpell* spell = m_spells.Item(sp)->GetData();
  if (player(spell->caster)->count_spell_questions() == 0 &&
      spell->n_questions == 0 && spell->delayed == 0) {
    if ((spell_list[ spell->id ].flags & SF_NO_TARGET) == 0 && spell->target < -1) {
      _add_target_question(spell->caster, spell->hand, sp);
      asking = true;
    }
    if ((spell_list[ spell->id ].flags & SF_MONSTER) && spell->detail < -1) {
      _add_new_monster_question(spell->caster, sp);
      asking = true;
    }
    if (spell->id == SPELL_SUMMON_ELEMENTAL && spell->detail < 0) {
      _add_elemental_question(spell->caster, sp);
      asking = true;
    }
    if ((spell->id == SPELL_PARALYSIS || spell->id == SPELL_CHARM_PERSON) &&
	spell->mind_hand == 0 && spell->target >= 0) {
      _add_hand_question(spell->caster, sp);
      asking = true;
    }
    if (spell->id == SPELL_CHARM_PERSON && spell->detail < 0 && spell->target >= 0) {
      _add_charm_question(spell->caster, sp);
      asking = true;
    }
  }
  return asking;
}


void Rules::_remove_permanency(int id) {
  Player* p = player(id);
  DoneSpellList::Node* node = NULL;
  DoneSpell* sp;

  for (node = m_spells.GetFirst() ; node ; node = node->GetNext()) {
    sp = node->GetData();
    if (sp->id == SPELL_PERMANENCY) break;
  }

  if (node) {
    // We can't delete the spell from the list or things explode
    // (numbering is important), so we just replace it with a spell
    // that does nothing.
    sp->id = SPELL_PLACEHOLDER;
    sp->hand = NEITHER_HAND;
  } else if (p->duration(D_PERMANENCY)) {
    p->set_duration(D_PERMANENCY, 0);
  } else {
    FAIL();
  }
}


void Rules::_remove_delayed_effect(int id) {
  Player* p = player(id);
  DoneSpellList::Node* node = NULL;
  DoneSpell* sp;

  for (node = m_spells.GetFirst() ; node ; node = node->GetNext()) {
    sp = node->GetData();
    if (sp->id == SPELL_DELAYED_EFFECT) break;
  }

  if (node) {
    // ditto.
    sp->id = SPELL_PLACEHOLDER;
    sp->hand = NEITHER_HAND;
  } else if (p->duration(D_DELAYED_EFFECT)) {
    p->set_duration(D_DELAYED_EFFECT, 0);
  } else {
    FAIL();
  }
}


void Rules::_ask_extra_questions(int id) {
  static const wxString noyes[2] = { wxT("No"), wxT("Yes") };
  DoneSpell* spells[ NUMBER_OF_HANDS + 1 ];
  DoneSpellList::Node* node = NULL;
  int n_spells;
  wxString s;
  Player* p = player(id);

  wxASSERT(p != NULL);

  // Ask the player if they want to release a delayed spell this turn.
  if (p->banked() >= 0 && !p->asked_banked()) {
    s.Printf(wxT("Do you want to release your delayed %s now?"), spell_list[ p->banked() ].name);
    Question* q = new Question(s, 2, noyes);
    q->set_type(QUESTION_USE_DELAYED);
    p->add_spell_question();
    p->set_asked_banked(true);
    _add_question(id, q);
  }

  // See whether this player has cast Permanency or Delayed Effect this turn.
  bool permanency = false, delayed = false;
  for (node = m_spells.GetFirst() ; node ; node = node->GetNext()) {
    if (node->GetData()->caster == id) {
      switch (node->GetData()->id) {
      case SPELL_PERMANENCY:      permanency = true; break;
      case SPELL_DELAYED_EFFECT:  delayed = true;    break;
      }
    }
  }

  // Ask the player which spell they want to make permanent.
  if (p->count_spell_questions() == 0 && !p->asked_perm() &&
      (permanency || p->duration(D_PERMANENCY))) {
    n_spells = 0;
    for (node = m_spells.GetFirst() ; node ; node = node->GetNext()) {
      if (node->GetData()->caster == id &&
	  (spell_list[ node->GetData()->id ].flags & SF_ENCHANTMENT) &&
	  !(spell_list[ node->GetData()->id ].flags & SF_NOPERM)) {
	spells[ n_spells++ ] = node->GetData();
      }
    }
    if (n_spells > 0) {
      wxString names[ n_spells + 1 ];
      for (int i = 0 ; i < n_spells ; i++) {
	names[i] = spell_list[ spells[i]->id ].name;
      }
      names[n_spells] = wxT("(No, not yet)");
      s = wxT("Do you want to make a spell permanent this round?");
      Question* q = new Question(s, n_spells + 1, names);
      q->set_type(QUESTION_PERMANENT);
      _add_question(id, q, true);
    }
    p->set_asked_perm(true);
  }

  // Ask the player which spell they want to delay.
  if (p->count_spell_questions() == 0 && !p->asked_delay() &&
      (delayed || p->duration(D_DELAYED_EFFECT))) {
    n_spells = 0;
    for (node = m_spells.GetFirst() ; node ; node = node->GetNext()) {
      if (node->GetData()->caster == id && node->GetData()->permanent == 0
	  && node->GetData()->id != SPELL_DELAYED_EFFECT &&
	  !(spell_list[ node->GetData()->id ].flags & SF_NOT_SPELL)) {
	spells[ n_spells++ ] = node->GetData();
      }
    }
    if (n_spells > 0) {
      wxString names[ n_spells + 1 ];
      for (int i = 0 ; i < n_spells ; i++) {
	names[i] = spell_list[ spells[i]->id ].name;
      }
      names[n_spells] = wxT("(No, not yet)");
      s = wxT("Do you want to store a spell with your Delayed Effect?");
      Question* q = new Question(s, n_spells + 1, names);
      q->set_type(QUESTION_DELAYED);
      player(id)->add_spell_question();
      _add_question(id, q, true);
    }
    p->set_asked_delay(true);
  }

  // Ask questions for all spells now, if the player's spell list is finished.
  int spell_num = 0;
  for (node = m_spells.GetFirst() ; node ; node = node->GetNext()) {
    DoneSpell* sp = node->GetData();
    if (sp->caster == id) {
      _ask_questions(spell_num);
    }
    spell_num++;
  }

  // Creatures don't act on hasted or time-stopped rounds.
  if (m_turn_type == TURN_NORMAL) {
    for (unsigned int i = 0 ; i < m_max_creatures ; i++) {
      Creature* c = creature(i);
      if (c && c->owner() == id && !c->asked() && c->is_alive() &&
	  c->type() != CREATURE_FIRE_EL && c->type() != CREATURE_ICE_EL) {
	_add_monster_question(id, i);
	c->set_asked();
      }
    }
  }

  // Send out the questions, in batches of MAX_QUESTIONS at a time.
  if (m_questions[id] != NULL) {
    m_batch_left[id] = m_questions[id]->GetCount() > MAX_QUESTIONS ?
      (size_t) MAX_QUESTIONS : m_questions[id]->GetCount();
    for (int x = 0 ; x < m_batch_left[id] ; x++) {
      Question* q = m_questions[id]->Item(x)->GetData();
      if (q->has_been_sent() == false) {
	g_server->send_event(id, q->to_event());
	q->set_sent();
      }
    }
  }
}


static void _exec_mind_control(Player* p) {
  wxASSERT(g_server != NULL);
  wxASSERT(p != NULL);
  int spell = -1;

  // Amnesia causes the target to repeat their last turn's gestures.
  if (p->duration(D_AMNESIA) > 0) {
    spell = D_AMNESIA;
    for (unsigned int x = 1 ; x < MAX_GESTURES ; x++) {
      if (p->get_gesture(LEFT_HAND, x) != GESTURE_ANTISPELL) {
	p->set_gesture(LEFT_HAND, p->get_gesture(LEFT_HAND, x));
	p->set_gesture(RIGHT_HAND, p->get_gesture(RIGHT_HAND, x));
	break;
      }
    }
    g_server->broadcast(msg_a_amnesia_hit(p));

  // Confusion causes the target to perform a random gesture (but not
  // nothing or stab) with a random hand.
  } else if (p->duration(D_CONFUSION) > 0) {
    spell = D_CONFUSION;
    int hand = (rand() % 2) ? LEFT_HAND : RIGHT_HAND;
    int gesture = GESTURE_NOTHING;
    while (gesture == GESTURE_NOTHING || gesture == GESTURE_KNIFE) {
      gesture = rand() % 8;
    }
    p->set_gesture(hand, gesture);
    g_server->broadcast(msg_a_confusion_hit(p, hand));

  // Charm Person causes a player to perform a specified gesture with
  // a specified hand, as chosen by the casting opponent.
  } else if (p->duration(D_CHARM_PERSON) > 0) {
    spell = D_CHARM_PERSON;
    p->set_gesture(p->mind_hand, p->mind_detail);
    g_server->broadcast(msg_a_charm_person_hit(p));

  // Fear causes the target to be unable to perform certain gestures.
  } else if (p->duration(D_FEAR) > 0) {
    spell = D_FEAR;
    int honk = 0;
    switch (p->get_gesture(LEFT_HAND, 0)) {
    case GESTURE_NOTHING:  case GESTURE_DIGIT:  case GESTURE_FINGERS:
    case GESTURE_SNAP:     case GESTURE_CLAP:
      p->set_gesture(LEFT_HAND, GESTURE_NOTHING);
      honk++;
    }
    switch (p->get_gesture(RIGHT_HAND, 0)) {
    case GESTURE_NOTHING:  case GESTURE_DIGIT:  case GESTURE_FINGERS:
    case GESTURE_SNAP:     case GESTURE_CLAP:
      p->set_gesture(RIGHT_HAND, GESTURE_NOTHING);
      honk += 2;
    }

    switch (honk) {
    case 0:  g_server->broadcast(msg_a_fear_miss(p));         break;
    case 1:  // fall through
    case 2:  g_server->broadcast(msg_a_fear_hit_1(p, honk));  break;
    case 3:  g_server->broadcast(msg_a_fear_hit_2(p));        break;
    default: FAIL();
    }

  // Paralysis causes one hand to repeat the last turn's gestures,
  // possibly altered a bit.
  } else if (p->duration(D_PARALYSIS) > 0) {
    spell = D_PARALYSIS;
    int gesture = GESTURE_ANTISPELL;
    for (unsigned int x = 1 ; x < MAX_GESTURES ; x++) {
      if (p->get_gesture(p->mind_hand, x) != GESTURE_ANTISPELL) {
	gesture = p->get_gesture(p->mind_hand, x);
	break;
      }
    }
    switch (gesture) {
    case GESTURE_WAVE:  p->set_gesture(p->mind_hand, GESTURE_PALM);    break;
    case GESTURE_SNAP:  p->set_gesture(p->mind_hand, GESTURE_DIGIT);   break;
    case GESTURE_CLAP:  p->set_gesture(p->mind_hand, GESTURE_FINGERS); break;
    default:            p->set_gesture(p->mind_hand, gesture);         break;
    }
    g_server->broadcast(msg_a_paralysis_hit(p));
  }
  
  // Mind-affecting spells attack hasted targets twice: once during
  // their hasted turn, and again during their next normal turn.
  if (spell >= 0) {
    if (p->duration(spell) == 99) {
      p->set_duration(spell, 0);
    } else if (p->duration(D_HASTE) > 0) {
      p->set_duration(spell, 99);
    }
  }
}


// I apologize for the atrocious hairiness of this function. I will
// attempt to ameliorate the unpleasantness with extra comments.

void Rules::start_questions() {
  // Alter the gesture lists to account for mind-affecting spells.
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    Player* p = player(i);
    if (p != NULL) {
      wxASSERT(p->count_spell_questions() == 0);
      p->set_done(true);
      if (p->active()) {
	_exec_mind_control(p);
      }
    }
  }

  // Now that the list of gestures has been finalized, record the
  // gestures for this turn in the log.
  g_window->log_hidden(wxT("\n\tTurn %d:"), m_turn);
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    Player* p = player(i);
    if (p != NULL) {
      g_window->log_hidden(wxT("%s (%d): %c %c"), C_STR(p->name()), p->health(),
			   gesture_to_char(p->get_gesture(LEFT_HAND, 0)),
			   gesture_to_char(p->get_gesture(RIGHT_HAND, 0)));
    }
  }
  g_window->log_hidden(wxT(""));

  m_spells.Clear();
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL && player(i)->active()) {
      // Now that we have all gestures, get the list of possible spells
      // formed by each player's hands. Count how many spells are formed
      // by each hand, and whether they have any both-hands spells.

      int both, hands[ NUMBER_OF_HANDS ];
      both = hands[ LEFT_HAND ] = hands[ RIGHT_HAND ] = 0;
      PossibleSpellList* list = player(i)->get_spells();
      for (PossibleSpellList::Node* node = list->GetFirst() ; node ;
	   node = node->GetNext()) {
	switch (node->GetData()->hand) {
	case LEFT_HAND:   hands[ LEFT_HAND ]++;  break;
	case RIGHT_HAND:  hands[ RIGHT_HAND ]++; break;
	case BOTH_HANDS:  both++;  break;
	default:          FAIL();
	}
      }

      if (player(i)->surrendered()) {
	_msg(i, msg_s_surrendered(), msg_o_surrendered(player(i)));
      }

      // If there are both-hands spells, that's a special case.
      if (both) {
	wxASSERT(hands[ LEFT_HAND ] == 0 || hands[ RIGHT_HAND ] == 0);

	// More than one? Ask which both-hand spell they meant.
	if (both > 1) {
	  _add_spell_question(i, list, BOTH_HANDS);

	// One both-hand spell?
	} else {
	  // If there are other possible single-hand spells, choose which.
	  if (hands[ LEFT_HAND ] > 0 || hands[ RIGHT_HAND ] > 0) {
	    int hand = hands[ LEFT_HAND ] ? LEFT_HAND : RIGHT_HAND;
	    _add_spell_question(i, list, hand, true);

	  } else {   // Otherwise, just use this both-hand spell.
	    _add_spell(i, BOTH_HANDS, list->GetFirst()->GetData()->id);
	  }
	}

      } else {  // No double-hand spells? OK, that's easy too.
	// for each hand...
	for (int x = 0 ; x < NUMBER_OF_HANDS ; x++) {
	  // how many spells can this hand choose from this round?
	  switch (hands[x]) {
	  case 0: break;  // None? Forget it.

	  // One? Pick it out of the list and set it as that hand's spell.
	  case 1: {
	    PossibleSpellList::Node* node;
	    for (node = list->GetFirst() ; node ; node = node->GetNext()) {
	      if (node->GetData()->hand == x) { break; }
	    }
	    wxASSERT(node);
	    _add_spell(i, x, node->GetData()->id);
	  } break;

	  // More than one? Ask the user which one they meant.
	  default:
	    _add_spell_question(i, list, x, false);
	    break;
	  }
	}
      }
      delete list;
    }
  }

  set_phase(ANSWERS_PHASE);
  _send_waiting_events();
  _send_gestures(true);
  _send_turn();

  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i)) {
      player(i)->set_asked_perm(false);
      player(i)->set_asked_delay(false);
      player(i)->set_asked_banked(false);
      _ask_extra_questions(i);
    }
  }
}


void Rules::add_answer(int id, const wxString& question, wxInt32 index) {
  wxASSERT(player(id) != NULL);
  if (m_questions[id] == NULL) {
    g_window->log(wxT("Spurious add_answer from player %d"), id);
    return;
  }

  Question* q = NULL;
  for (QueryList::Node* node = m_questions[id]->GetFirst() ; node ; node = node->GetNext()) {
    if (node->GetData()->question() == question) {
      q = node->GetData();
      break;
    }
  }
  if (q == NULL) {
    g_window->log(wxT("Bogus add_answer question from player %d"), id);
    return;
  }
  if (index < 0 || index >= q->answer_count()) {
    g_window->log(wxT("Bogus add_answer index from player %d"), id);
    return;
  }

  DoneSpell* sp = NULL;
  if (q->spell() >= 0) {
    sp = m_spells.Item(q->spell())->GetData();
    sp->n_questions--;
    wxASSERT(id == sp->caster);
  }

  switch (q->type()) {
  case QUESTION_SPELL:
    for (int i = 0 ; i < NUMBER_OF_SPELLS; i++) {
      if (q->answer(index) == spell_list[i].name) {
	sp->id = i;
	if (spell_list[i].offhand[0] != '\0' && spell_list[i].offhand[0] != '?') {
	  sp->hand = BOTH_HANDS;
	}
      }
    }
    player(id)->remove_spell_question();
    break;

  case QUESTION_TARGET:
    if (q->answer(index) == wxT("(nobody)")) {
      sp->target = -1;
    } else {
      for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
	if (being(i) != NULL && q->answer(index) == being(i)->name()) {
	  sp->target = i;
	  break;
	}
      }
      wxASSERT(sp->target >= 0);
      switch (sp->id) {
      case SPELL_CHARM_MONSTER:
	being(sp->target)->mind_detail = sp->caster;
	break;
      }
    }
    break;

  case QUESTION_NEW_MONSTER:
    if (q->answer(index) == wxT("(nobody)")) {
      sp->detail = -1;
    } else {
      for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
	if (being(i) != NULL && q->answer(index) == being(i)->name()) {
	  sp->detail = i;
	  break;
	}
      }
      wxASSERT(sp->detail >= 0);
    }
    sp->target = sp->caster;
    break;

  case QUESTION_MONSTER:
    for (unsigned int i = 0 ; i < MAX_BEINGS ; i++) {
      if (being(i) != NULL && q->answer(index) == being(i)->name()) {
	creature(q->monster())->set_target(i);
	break;
      } else {
	creature(q->monster())->set_target(-1);
      }
    }
    break;

  case QUESTION_ELEMENTAL:
    sp->detail = (q->answer(index) == wxT("Fire")) ? CREATURE_FIRE_EL : CREATURE_ICE_EL;
    break;

  case QUESTION_USE_DELAYED:
    if (q->answer(index) == wxT("Yes")) {
      _add_spell(id, DELAYED_EFFECT, player(id)->banked());
      player(id)->set_banked(-1);
    }
    player(id)->remove_spell_question();
    break;

  case QUESTION_PERMANENT:
    for (DoneSpellList::Node* node = m_spells.GetFirst() ; node ; node = node->GetNext()) {
      if (q->answer(index) == spell_list[ node->GetData()->id ].name &&
	  node->GetData()->caster == id) {
	node->GetData()->permanent = 1;
	_remove_permanency(id);
      }
    }
    break;

  case QUESTION_DELAYED:
    for (DoneSpellList::Node* node = m_spells.GetFirst() ; node ; node = node->GetNext()) {
      if (q->answer(index) == spell_list[ node->GetData()->id ].name &&
	  node->GetData()->caster == id) {
	node->GetData()->delayed = 1;
	_remove_delayed_effect(id);
      }
    }
    player(id)->remove_spell_question();
    break;

  case QUESTION_HAND:
    sp->mind_hand = (index == 0 ? LEFT_HAND : RIGHT_HAND) + 1;
    break;

  case QUESTION_CHARM:
    sp->detail = index + 1;
    break;

  default:  FAIL();
  }

  m_batch_left[id]--;
  m_questions[id]->DeleteObject(q); 
  _ask_extra_questions(id);
  if (m_questions[id]->GetCount() == 0) {
    player(id)->set_done(true);
    delete m_questions[id];
    m_questions[id] = NULL;
  }
  _send_waiting_events();
}


bool Rules::phase_is_done() {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    Player* p = player(i);
    if (p != NULL && p->active() && p->done() == false) {
      return false;
    }
  }
  return true;
}


void Rules::_send_waiting_events() {
  wxUint32 vec = 0;

  wxASSERT(g_server != NULL);
  wxASSERT(MAX_PLAYERS <= 32);

  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL && player(i)->done()) {
      vec |= (1 << i);
    }
  }
  g_server->broadcast(Event::Waiting(vec));
}


void Rules::_msg(int you, const Event& smsg) const {
  wxASSERT(g_server != NULL);
  g_server->send_event(you, smsg);
}


void Rules::_msg(int you, const Event& smsg, const Event& omsg) const {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL) {
      g_server->send_event(i, (int) i == you ? smsg : omsg);
    }
  }
}


void Rules::_msg(int you, const Event& smsg, const Event& omsg,
		      int target, const Event& tmsg) const {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL) {
      if ((int) i == you) {
	g_server->send_event(i, smsg);
      } else if ((int) i == target) {
	g_server->send_event(i, tmsg);
      } else {
	g_server->send_event(i, omsg);
      }
    }
  }
}

