/* $Id: Messages.cpp,v 1.3 2008/04/12 21:20:44 dpt Exp $
 *
 * Messages.cpp: Functions that generate custom Event::Msg objects.
 */

#include "Spellcast.h"


#define NO_PARAMETERS(func, string)    \
  Event func () {                      \
    return Event::Msg(wxT(string));    \
  }

#define BEING_NAME(func, string)					\
  Event func (Being* b) {						\
    wxASSERT(b != NULL);						\
    return Event::Msg(wxT(string), C_STR(b->name()));			\
  }

#define PLAYER_NAME(func, string)					\
  Event func (Player* p) {						\
    wxASSERT(p != NULL);						\
    return Event::Msg(wxString::Format(wxT(string), C_STR(p->name()))); \
  }

#define CREATURE_NAME(func, string)					\
  Event func (Creature* c) {						\
    wxASSERT(c != NULL);						\
    return Event::Msg(wxString::Format(wxT(string), C_STR(c->name()))); \
  }

#define CREATURE_AND_BEING(func, string)				\
  Event func (Creature* c, Being* b) {					\
    wxASSERT(c != NULL && b != NULL);					\
    return Event::Msg(wxString::Format(wxT(string), C_STR(c->name()), C_STR(b->name()))); \
  }

#define SPELL_AND_BEING(func, string)					\
  Event func (Being* b, DoneSpell* sp) {				\
    wxASSERT(b != NULL && sp != NULL);					\
    return Event::Msg(wxString::Format(wxT(string), spell_list[ sp->id ].name, C_STR(b->name()))); \
  }

#define SPELL_ID_AND_BEING(func, string)				\
  Event func (Being* b, int spell) {					\
    wxASSERT(b != NULL && spell >= 0 && spell < NUMBER_OF_SPELLS);	\
    return Event::Msg(wxString::Format(wxT(string), spell_list[ spell ].name, C_STR(b->name()))); \
  }

#define SPELL_NAME(func, string)					\
  Event func (DoneSpell* sp) {						\
    wxASSERT(sp != NULL);						\
    return Event::Msg(wxString::Format(wxT(string), spell_list[ sp->id ].name)); \
  }

#define SPELL_ID_NAME(func, string)					\
  Event func (int spell) {						\
    wxASSERT(spell >= 0 && spell < NUMBER_OF_SPELLS);			\
    return Event::Msg(wxString::Format(wxT(string), spell_list[ spell ].name)); \
  }

#define BEING_HIS(func, string)						\
  Event func (Being* b) {						\
    wxASSERT(b != NULL);						\
    return Event::Msg(wxString::Format(wxT(string), C_STR(b->name()), b->his())); \
  }


static const char* number(int num) {
  static char buf[6];

  switch (num) {
  case 0: return "zero";
  case 1: return "one";
  case 2: return "two";
  case 3: return "three";
  case 4: return "four";
  case 5: return "five";
  case 6: return "six";
  case 7: return "seven";
  case 8: return "eight";
  case 9: return "nine";
  }
  sprintf(buf, "%d", num);
  return buf;
}


static const char* space_number(int num) {
  static char buf[7];
  sprintf(buf, " %s", number(num));
  return buf;
}


NO_PARAMETERS(msg_s_game_start, "You advance confidently into the arena. The referee casts the formal Dispel Magic and Anti-Spell on you...");
NO_PARAMETERS(msg_s_dispel_counter, "The Counter-Spell on you is dispelled.");
NO_PARAMETERS(msg_s_dispel_mirror, "The Magic Mirror on you is dispelled.");
NO_PARAMETERS(msg_s_got_counter, "Your magical senses go numb as a Counter-Spell surrounds you.");
NO_PARAMETERS(msg_s_got_mirror, "The bright haze of a Magic Mirror surrounds you.");
NO_PARAMETERS(msg_s_counter_mirror, "The Magic Mirror on you is destroyed by the Counter-Spell.");
NO_PARAMETERS(msg_s_surrendered, "Oh dear. You seem to have surrendered.");
NO_PARAMETERS(msg_s_arena_full, "The arena is packed wall-to-wall with monsters and corpses! There's not enough room for your summoning spell to work.");
NO_PARAMETERS(msg_a_draw_game, " *** The game is a draw! *** ");
NO_PARAMETERS(msg_a_everyone_dead, "Everyone is dead!");
NO_PARAMETERS(msg_a_survivors_surrender, "The survivors have all surrendered to each other!");
NO_PARAMETERS(msg_a_everyone_surrenders, "Everyone has surrendered to each other!");
NO_PARAMETERS(msg_s_start_haste, "Everyone else begins moving very slowly.");
NO_PARAMETERS(msg_s_continue_haste, "Everyone else is still moving slowly.");
NO_PARAMETERS(msg_s_end_haste, "Everyone comes back up to normal speed.");
NO_PARAMETERS(msg_s_start_timestop, "Everything around you stops dead. The world is silent and motionless.");
NO_PARAMETERS(msg_s_end_timestop, "Everything begins moving again.");
NO_PARAMETERS(msg_s_has_died, "You have died!");
NO_PARAMETERS(msg_s_permanency_done, "The Permanency spell hovering around you dies away.");
NO_PARAMETERS(msg_s_delayed_effect_done, "The Delayed Effect spell hovering around you dies away.");
NO_PARAMETERS(msg_s_lb2_failed, "Since you have already cast Lightning Bolt with that formulation, the spell fizzles.");
NO_PARAMETERS(msg_a_dispel_magic, "The magical energies in the arena fade away.");
NO_PARAMETERS(msg_s_summon_fire_el, "A blazing Fire Elemental materializes above you.");
NO_PARAMETERS(msg_s_summon_ice_el, "A glittering Ice Elemental materializes above you.");
NO_PARAMETERS(msg_a_elementals_fight, "The Fire and Ice Elementals tear into each other, and whirl into a column of steam that dissipates in moments.");
NO_PARAMETERS(msg_a_storms_fight, "The Fire and Ice Storms tear each other into shreds of harmless vapor.");
NO_PARAMETERS(msg_a_storms_and_elementals, "The Fire and Ice Elementals whirl into each other, screaming in fury, and the Fire and Ice storms shriek in response. The column of Elemental energy rises to meet the cloud of vapor that descends over the arena. You are buffeted by the conflicting energies; but within moments, nothing remains.");
NO_PARAMETERS(msg_s_fod_raise_cancel, "For a moment, you feel a sudden pressure in your chest, as if an iron band was squeezing your heart. Then it fades, as the Raise Dead spell burns through you.");
NO_PARAMETERS(msg_s_fb_storm_cancel, "The Fireball hurtling towards you is snuffed by the Ice Storm, leaving you enveloped in warm steam.");
NO_PARAMETERS(msg_s_got_resist_heat, "The Resist Heat wraps you in a pleasant blue coolness.");
NO_PARAMETERS(msg_s_got_resist_cold, "The Resist Cold wraps you in a pleasant pink warmth.");
NO_PARAMETERS(msg_s_more_resist_heat, "Your veil of Resist Heat continues to glow coolly." );
NO_PARAMETERS(msg_s_more_resist_cold, "Your veil of Resist Heat continues to glow warmly.");
NO_PARAMETERS(msg_s_got_prot_evil, "A white circle of Protection from Evil springs up around you. You feel the Shield aura cloaking your skin.");
NO_PARAMETERS(msg_s_more_prot_evil, "Your circle of Protection from Evil flares back to full strength.");
NO_PARAMETERS(msg_s_fade_prot_evil_3, "Your circle of Protection from Evil is beginning to fade.");
NO_PARAMETERS(msg_s_fade_prot_evil_2, "Your circle of Protection from Evil is dimmer now.");
NO_PARAMETERS(msg_s_fade_prot_evil_1, "Your circle of Protection from Evil is nearly gone.");
NO_PARAMETERS(msg_s_anti_rmench_cancel, "The Anti-Spell fizzes away into nothing as it approaches you.");
NO_PARAMETERS(msg_s_antispell_hit, "The Anti-Spell jolts you, interrupting the flow of magical energy in your hands.");
NO_PARAMETERS(msg_s_delay_rmench_cancel, "The Delayed Effect fizzes away into nothing as it approaches you.");
NO_PARAMETERS(msg_s_delay_hit, "The Delayed Effect spreads out in a cloud around you, waiting for you to complete a spell.");
NO_PARAMETERS(msg_s_perm_rmench_cancel, "The Permanency fizzes away into nothing as it approaches you.");
NO_PARAMETERS(msg_s_perm_hit, "The Permanency spreads out in a cloud around you, waiting for you to complete a spell.");
NO_PARAMETERS(msg_s_blind_hit, "A haze suddenly appears before your eyes. The world dims rapidly, and you are shortly enveloped in total darkness.");
NO_PARAMETERS(msg_s_blind_fades, "The dark veil falls away from your eyes.");
NO_PARAMETERS(msg_s_blind_still, "You are still blind.");
NO_PARAMETERS(msg_s_invis_hit, "The world becomes strangely misted in front of your eyes. Looking down, you see that your hands are no longer visible.");
NO_PARAMETERS(msg_s_invis_fades, "Your limbs have become visible again.");
NO_PARAMETERS(msg_s_invis_still, "You are still invisible.");
NO_PARAMETERS(msg_s_rmench_hit, "The Remove Enchantment whirls its dissipative energies around you.");
NO_PARAMETERS(msg_s_dispel_heat, "Your blue aura of Resist Heat fades away.");
NO_PARAMETERS(msg_s_dispel_cold, "Your pink aura of Resist Cold fades away.");
NO_PARAMETERS(msg_s_dispel_disease, "Your fever suddenly abates. You feel much better.");
NO_PARAMETERS(msg_s_dispel_poison, "The numbness and heaviness falls from your limbs. You feel much better.");
NO_PARAMETERS(msg_s_dispel_delay, "The Delayed Effect spell hovering around you disintegrates.");
NO_PARAMETERS(msg_s_dispel_perm, "The Permanency spell hovering around you disintegrates.");
NO_PARAMETERS(msg_s_dispel_prot, "Your circle of Protection flickers out.");
NO_PARAMETERS(msg_s_dispel_haste, "Everyone comes back up to normal speed.");
NO_PARAMETERS(msg_s_cancel_haste, "The Haste on you is nullified.");
NO_PARAMETERS(msg_s_cancel_timestop, "The Time Stop on you is nullified.");
NO_PARAMETERS(msg_s_dispel_invis, "Your body tingles and becomes visible again.");
NO_PARAMETERS(msg_s_dispel_blind, "Sight bursts in upon you again.");
NO_PARAMETERS(msg_s_charmed_still, "You are still Charmed.");
NO_PARAMETERS(msg_s_fire_storm_resist, "The Fire Storm washes over you, but you feel only a gentle warmth.");
NO_PARAMETERS(msg_s_ice_storm_resist, "The Ice Storm washes over you, but you feel only a gentle coolness.");
NO_PARAMETERS(msg_s_fod_hit, "You feel a sudden pressure in your chest, as if an iron band was squeezing your heart. You fall to your knees as the world dims...");
NO_PARAMETERS(msg_s_raise_dead, "You are suddenly aware of a blast of energy running down every nerve! This is something of a shock, since you had just gotten used to being dead. Oh, well -- you'll be back eventually.");
NO_PARAMETERS(msg_s_all_healed, "You are in perfect health again.");


BEING_NAME(msg_o_dispel_counter, "The Counter-Spell on %s is dispelled.");
BEING_NAME(msg_o_dispel_mirror, "The Magic Mirror on %s is dispelled.");
BEING_NAME(msg_o_got_counter, "A Counter-Spell flares around %s.");
BEING_NAME(msg_o_got_mirror, "A Magic Mirror swirls around %s.");
BEING_NAME(msg_o_counter_mirror, "The Magic Mirror on %s is destroyed by the Counter-Spell.");
BEING_NAME(msg_a_dispel_creature, "The Dispel Magic starts to tear %s apart.");
BEING_NAME(msg_a_dispel_corpse, "The corpse of %s disintegrates, destroyed by the Dispel Magic.");
BEING_NAME(msg_o_start_haste, "%s begins moving very quickly.");
BEING_NAME(msg_o_continue_haste, "%s is still moving quickly.");
BEING_NAME(msg_o_end_haste, "%s slows down to normal speed.");
BEING_NAME(msg_o_start_timestop, "%s accelerates into a flickering blur.");
BEING_NAME(msg_o_end_timestop, "%s reappears, moving at normal speed.");
BEING_NAME(msg_o_has_died, "%s has died!");
BEING_NAME(msg_o_fod_raise_cancel, "The Finger of Death and Raise Dead spells aimed at %s cancel each other.");
BEING_NAME(msg_o_got_resist_heat, "The Resist Heat shrouds %s in a cool blue veil.");
BEING_NAME(msg_o_got_resist_cold, "The Resist Cold shrouds %s in a warm pink veil.");
BEING_NAME(msg_o_more_resist_heat, "%s's veil of Resist Heat continues to glow coolly.");
BEING_NAME(msg_o_more_resist_cold, "%s's veil of Resist Heat continues to glow warmly.");
BEING_NAME(msg_o_got_prot_evil, "A white circle of Protection from Evil springs up around %s.");
BEING_NAME(msg_o_more_prot_evil, "%s's circle of Protection from Evil flares back to full strength.")
;
BEING_NAME(msg_o_fade_prot_evil_3, "%s's circle of Protection from Evil is beginning to fade.");
BEING_NAME(msg_o_fade_prot_evil_2, "%s's circle of Protection from Evil is dimmer now.");
BEING_NAME(msg_o_fade_prot_evil_1, "%s's circle of Protection from Evil is nearly gone.");
BEING_NAME(msg_o_anti_rmench_cancel, "The Anti-Spell fizzes away into nothing as it approaches %s.");
BEING_NAME(msg_a_antispell_miss, "The Anti-Spell whizzes through %s without any effect.");
BEING_NAME(msg_o_delay_rmench_cancel, "The Delayed Effect fizzes away into nothing as it approaches %s.");
BEING_NAME(msg_o_delay_hit, "The Delayed Effect spreads out around %s.");
BEING_NAME(msg_a_delay_miss, "The Delayed Effect whizzes through %s without any effect.");
BEING_NAME(msg_o_perm_rmench_cancel, "The Permanency fizzes away into nothing as it approaches %s.");
BEING_NAME(msg_o_perm_hit, "The Permanency spreads out around %s.");
BEING_NAME(msg_a_perm_miss, "The Permanency whizzes through %s without any effect.");
BEING_NAME(msg_o_blind_fades, "The milky cast fades from %s's eyes.");
BEING_NAME(msg_o_blind_still, "%s is still blind.");
BEING_NAME(msg_o_invis_hit, "%s turns quietly translucent, and fades from view.");
BEING_NAME(msg_o_invis_still, "%s is still invisible.");
BEING_NAME(msg_o_rmench_hit, "The Remove Enchantment whirls its dissipative energies around %s");
BEING_NAME(msg_o_dispel_heat, "%s's blue aura of Resist Heat fades away.");
BEING_NAME(msg_o_dispel_cold, "%s's pink aura of Resist Cold fades away.");
BEING_NAME(msg_o_dispel_poison, "%s's color improves dramatically.");
BEING_NAME(msg_o_dispel_delay, "The Delayed Effect spell hovering around %s disintegrates.");
BEING_NAME(msg_o_dispel_perm, "The Permanency spell hovering around %s disintegrates.");
BEING_NAME(msg_o_dispel_prot, "%s's circle of Protection flickers out.");
BEING_NAME(msg_o_dispel_haste, "%s slows down to normal speed.");
BEING_NAME(msg_o_cancel_haste, "The Haste on %s is nullified.");
BEING_NAME(msg_o_cancel_timestop, "The Time Stop on %s is nullified.");
BEING_NAME(msg_o_dispel_invis, "%s suddenly becomes visible again.");
BEING_NAME(msg_o_dispel_blind, "%s's eyes abruptly clear.");
BEING_NAME(msg_o_charmed_still, "%s is still Charmed.");
BEING_NAME(msg_o_all_healed, "%s is in perfect health again.");


PLAYER_NAME(msg_a_fear_hit_2, "Terrified, %s does nothing with either hand.");
PLAYER_NAME(msg_o_surrendered, "%s makes the gesture of surrender!");
PLAYER_NAME(msg_o_arena_full, "The arena is packed wall-to-wall with monsters and corpses! There's not enough room for %s's summoning spell to work.");
PLAYER_NAME(msg_a_sole_survivor, "%s is the sole survivor!");
PLAYER_NAME(msg_a_who_won, " *** %s has won! *** ");
PLAYER_NAME(msg_a_nobody_left, "%s finds that there is nobody left to surrender to!");
PLAYER_NAME(msg_o_permanency_done, "The Permanency spell hovering around %s dies away.");
PLAYER_NAME(msg_o_delayed_effect_done, "The Delayed Effect spell hovering around %s dies away.");
PLAYER_NAME(msg_o_lb2_failed, "%s's Lightning Bolt fizzles.");
PLAYER_NAME(msg_o_summon_fire_el, "A blazing Fire Elemental materializes above %s.");
PLAYER_NAME(msg_o_summon_ice_el, "A glittering Ice Elemental materializes above %s.");


CREATURE_NAME(msg_a_monster_paralyzed, "%s is paralyzed and cannot attack.");
CREATURE_NAME(msg_a_monster_amnesia_2, "%s stares blankly into the distance.");
CREATURE_NAME(msg_s_monster_cant_see, "%s tries to attack you, but cannot see you.");
CREATURE_NAME(msg_s_monster_blocked, "The attack of %s is blocked by your Shield.");
CREATURE_NAME(msg_s_monster_fireblock, "%s reaches towards you, but you feel only a warm breeze.");
CREATURE_NAME(msg_s_monster_firethrow, "%s hurls tongues of fire at you.");
CREATURE_NAME(msg_s_monster_iceblock, "%s reaches towards you, but you feel only a chill breeze.");
CREATURE_NAME(msg_s_monster_icethrow, "%s hurls splinters of ice at you.");
CREATURE_NAME(msg_s_monster_attacks, "%s attacks you.");
CREATURE_NAME(msg_a_fire_el_resisted, "%s gutters and flickers out under the influence of the Resist Heat.");
CREATURE_NAME(msg_a_ice_el_resisted, "%s melts away under the influence of the Resist Cold.");
CREATURE_NAME(msg_a_ice_el_fireballed, "%s is vaporized by the Fireball.");
CREATURE_NAME(msg_a_rmench_stress, "The Remove Enchantment starts to tear %s apart.");
CREATURE_NAME(msg_a_paralyzed_still, "%s is still afflicted with Paralysis.");


CREATURE_AND_BEING(msg_a_monster_confused, "Overcome by confusion, %s strikes at %s");
CREATURE_AND_BEING(msg_o_monster_firethrow, "%s hurls fire at %s.");
CREATURE_AND_BEING(msg_o_monster_blocked, "The attack of %s is blocked by %s's Shield.");
CREATURE_AND_BEING(msg_o_monster_icethrow, "%s hurls ice at %s.");
CREATURE_AND_BEING(msg_o_monster_attacks, "%s attacks %s.");
CREATURE_AND_BEING(msg_a_charmed_still, "%s is still enamoured of %s.");


SPELL_AND_BEING(msg_o_countered, "The %s is destroyed by the Counter-Spell around %s.");
SPELL_AND_BEING(msg_s_two_mirrors, "Your %s is reflected back and forth between %s's Magic Mirror and your own! It rapidly decays and dissipates.");
SPELL_AND_BEING(msg_s_mirrored, "Your %s reflects from %s's Magic Mirror back at you.");
SPELL_AND_BEING(msg_o_bounce_countered, "The reflected %s is destroyed by the Counter-Spell around %s.");


SPELL_ID_AND_BEING(msg_o_mind_curl, "The %s spell curls around %s's mind.");
SPELL_ID_AND_BEING(msg_o_mind_fail, "The %s spell swirls around %s's mind, but cannot sink in.");
SPELL_ID_AND_BEING(msg_a_spell_stress, "Under the stress of the %s spell, %s shivers and disintegrates into random energies.");


SPELL_NAME(msg_a_dispelled, "The %s is dispelled.");
SPELL_NAME(msg_s_countered, "The %s is destroyed by your Counter-Spell.");
SPELL_NAME(msg_s_made_permanent, "Your Permanency spell wraps itself around your %s spell.");
SPELL_NAME(msg_s_made_delayed, "You cast %s; the spell is caught in the web of your Delayed Effect. It rises to hover by your head.");
SPELL_NAME(msg_s_bounce_countered, "The reflected %s is destroyed by your Counter-Spell.");


SPELL_ID_NAME(msg_s_mind_curl, "The %s spell curls around your mind.");
SPELL_ID_NAME(msg_s_mind_fail, "The %s spell swirls around your mind, but cannot sink in.");


BEING_HIS(msg_o_blind_hit, "%s blinks, then stumbles slightly as %s eyes turn milky.");
BEING_HIS(msg_o_dispel_disease, "%s seems to have stopped shivering, and %s skin is no longer flushed and damp.");
BEING_HIS(msg_o_fire_storm_resist, "The Fire Storm washes over %s, who remains unharmed inside %s blue aura.");
BEING_HIS(msg_o_ice_storm_resist, "The Ice Storm washes over %s, who remains unharmed inside %s pink aura.");
BEING_HIS(msg_o_antispell_hit, "The Anti-Spell jolts %s, causing the magical energies around %s hands to flicker.");




Event msg_o_game_start(Player* p) {
  wxASSERT(p != NULL);
  return Event::Msg(wxT("%s strides defiantly into the arena. The referee casts the formal Dispel Magic and Anti-Spell on %s..."),
		    C_STR(p->name()), p->him());
}


Event msg_a_amnesia_hit(Player* p) {
  wxASSERT(p != NULL);
  return Event::Msg(wxT("In a fit of absent-mindedness, %s repeats %s gestures from last turn: %s with the left hand, %s with the right."),
		    C_STR(p->name()), p->his(),
		    gesture_name(p->get_gesture(LEFT_HAND, 0)),
		    gesture_name(p->get_gesture(LEFT_HAND, 0)));
}


Event msg_a_confusion_hit(Player* p, int hand) {
  wxASSERT(p != NULL);
  return Event::Msg(wxT("%s accidentally makes a %s with %s %s hand."),
		    C_STR(p->name()), gesture_name(p->get_gesture(hand, 0)), p->his(),
		    hand == LEFT_HAND ? "left" : "right");
}


Event msg_a_charm_person_hit(Player* p) {
  wxASSERT(p != NULL);
  return Event::Msg(wxT("%s mysteriously makes a %s with %s %s hand."),
		    C_STR(p->name()), gesture_name(p->mind_detail), p->his(),
		    p->mind_hand == LEFT_HAND ? "left" : "right");
}


Event msg_a_fear_miss(Player* p) {
  wxASSERT(p != NULL);
  return Event::Msg(wxT("%s is terrified, but manages %s gestures anyway."),
		    C_STR(p->name()), p->his());
}


Event msg_a_fear_hit_1(Player* p, int hand) {
  wxASSERT(p != NULL);
  return Event::Msg(wxT("Terrified, %s does nothing with %s %s hand."),
		    C_STR(p->name()), p->his(), hand == 1 ? "left" : "right");
}


Event msg_a_paralysis_hit(Player* p) {
  wxASSERT(p != NULL);
  if (p->get_gesture(p->mind_hand, 0) == GESTURE_NOTHING) {
    return Event::Msg(wxT("%s's frozen %s hand does nothing this turn."),
		      C_STR(p->name()), p->mind_hand == LEFT_HAND ? "left" : "right");
  }
  return Event::Msg(wxT("%s's %s hand is frozen in a %s."),
		    C_STR(p->name()), p->mind_hand == LEFT_HAND ? "left" : "right",
		    gesture_name(p->get_gesture(p->mind_hand, 0)));
}


Event msg_a_monster_amnesia_1(Creature* c, Being* t) {
  wxASSERT(c != NULL && t != NULL);
  return Event::Msg(wxT("%s strikes at %s with a blank expression on %s face."),
		    C_STR(c->name()), C_STR(t->name()), c->his());
}


Event msg_a_monster_not_self(Creature* c) {
  wxASSERT(c != NULL);
  return Event::Msg(wxT("%s refuses to attack %sself."), C_STR(c->name()), c->him());
}


Event msg_o_monster_cant_see(Creature* c, Being* t) {
  wxASSERT(c != NULL && t != NULL);
  return Event::Msg(wxT("%s tries to attack %s, but cannot see %s."),
		    C_STR(c->name()), C_STR(t->name()), t->him());
}


Event msg_o_monster_fireblock(Creature* c, Being* t) {
  wxASSERT(c != NULL && t != NULL);
  return Event::Msg(wxT("%s reaches toward %s, but cannot penetrate %s blue aura."),
		    C_STR(c->name()), C_STR(t->name()), t->his());
}


Event msg_o_monster_iceblock(Creature* c, Being* t) {
  wxASSERT(c != NULL && t != NULL);
  return Event::Msg(wxT("%s reaches toward %s, but cannot penetrate %s pink aura."),
		    C_STR(c->name()), C_STR(t->name()), t->his());
}


Event msg_a_win_by_surrender(Player* p, Player* stiff = NULL) {
  wxASSERT(p != NULL);
  if (stiff) {
    return Event::Msg(wxT("%s has surrendered to %s!"), C_STR(stiff->name()), C_STR(p->name()));
  }
  return Event::Msg(wxT("All of %s's opponents have surrendered!"), C_STR(p->name()));
}


Event msg_o_disease_progress(Being* b, int time) {
  wxString s;
  switch (time) {
  case 6:
    s.Printf(wxT("%s sneezes loudly."), C_STR(b->name()));
    break;
  case 5:
    s.Printf(wxT("%s starts to look hot and flushed."), C_STR(b->name()));
    break;
  case 4:
    s.Printf(wxT("%s looks even more flushed."), C_STR(b->name()));
    break;
  case 3:
    s.Printf(wxT("%s is flushed and sweating, and seems somewhat unsteady on %s feet."), C_STR(b->name()), b->his());
    break;
  case 2:
    s.Printf(wxT("%s looks flushed and is shivering violently."), C_STR(b->name()));
    break;
  case 1:
    s.Printf(wxT("%s crumples to the ground, agony on %s sweating features."), C_STR(b->name()), b->his());
    break;
  default:  FAIL();
  }
  return Event::Msg(s);
}


Event msg_s_disease_progress(int time) {
  wxString s;
  switch (time) {
  case 6:
    s = wxT("You sneeze loudly.");
    break;
  case 5:
    s = wxT("You begin to feel somewhat feverish.");
    break;
  case 4:
    s = wxT("Your fever is growing worse.");
    break;
  case 3:
    s = wxT("You are very hot and somewhat dizzy, and your bones are starting to ache.");
    break;
  case 2:
    s = wxT("You are feverish and shivering at the same time, and you ache all over.");
    break;
  case 1:
    s = wxT("Your knees give way and you fall to the ground. Your head pounds unmercifully as consciousness slips away...");
    break;
  default:  FAIL();
  }
  return Event::Msg(s);
}


Event msg_o_poison_progress(Being* b, int time) {
  wxString s;
  switch (time) {
  case 6:
    s.Printf(wxT("%s suddenly looks somewhat pale."), C_STR(b->name()));
    break;
  case 5:
    s.Printf(wxT("%s is looking pale."), C_STR(b->name()));
    break;
  case 4:
    s.Printf(wxT("%s is looking very pale and weak."), C_STR(b->name()));
    break;
  case 3:
    s.Printf(wxT("%s is white as a sheet, and seems a little unsteady."), C_STR(b->name()));
    break;
  case 2:
    s.Printf(wxT("%s is having difficulty staying on %s feet."), C_STR(b->name()), b->his());
    break;
  case 1:
    s.Printf(wxT("%s collapses in a heap, twitching slightly."), C_STR(b->name()));
    break;
  default:  FAIL();
  }
  return Event::Msg(s);
}


Event msg_s_poison_progress(int time) {
  wxString s;
  switch (time) {
  case 6:
    s = wxT("A flash of pain lances along your bones, and then fades away.");
    break;
  case 5:
    s = wxT("Your toes suddenly feel slightly numb.");
    break;
  case 4:
    s = wxT("Your hands and feet are getting numb.");
    break;
  case 3:
    s = wxT("Your limbs are heavy, and your fingers are getting clumsy.");
    break;
  case 2:
    s = wxT("You cannot feel your hands and feet, and your vision is becoming clouded.");
    break;
  case 1:
    s = wxT("You stumble and fall to your knees, vision dimming. Numbness creeps through your bones, and you cannot tell if you are still breathing...");
    break;
  default:  FAIL();
  }
  return Event::Msg(s);
}


Event msg_o_visibility_miss(Player* c, Being* t, DoneSpell* sp) {
  wxASSERT(c != NULL && t != NULL);
  wxASSERT(sp != NULL);
  return Event::Msg(wxT("%s is unable to see %s; %s %s goes wild."),
		    C_STR(c->name()), C_STR(t->name()), c->his(), spell_list[ sp->id ].name);
}


Event msg_s_visibility_miss(Being* t, DoneSpell* sp) {
  wxASSERT(t != NULL && sp != NULL);
  return Event::Msg(wxT("You are unable to see %s; your %s goes wild."),
		    C_STR(t->name()), spell_list[ sp->id ].name);
}


Event msg_t_visibility_miss(Player* c, DoneSpell* sp) {
  wxASSERT(c != NULL && sp != NULL);
  return Event::Msg(wxT("%s is unable to see you; %s %s goes wild."),
		    C_STR(c->name()), c->his(), spell_list[ sp->id ].name);
}


Event msg_o_made_delayed(Player* p, DoneSpell* sp) {
  wxASSERT(p != NULL && sp != NULL);
  return Event::Msg(wxT("%s casts %s; the spell is caught in the web of %s Delayed Effect."),
		    C_STR(p->name()), spell_list[ sp->id ].name, p->his());
}


Event msg_o_two_mirrors(Player* p, Being* b, DoneSpell* sp) {
  wxASSERT(p != NULL && b != NULL && sp != NULL);
  return Event::Msg(wxT("%s's %s is reflected back and forth between %s's Magic Mirror and %s own! It rapidly decays and dissipates."),
		    C_STR(p->name()), spell_list[ sp->id ].name, C_STR(b->name()), p->his());
}


Event msg_t_two_mirrors(Player* p, DoneSpell* sp) {
  wxASSERT(p != NULL && sp != NULL);
  return Event::Msg(wxT("%s's %s is reflected back and forth between your Magic Mirror and %s own! It rapidly decays and dissipates."), C_STR(p->name()),
		    spell_list[ sp->id ].name, p->his());
}

Event msg_o_mirrored(Player* p, Being* b, DoneSpell* sp) {
  wxASSERT(p != NULL && b != NULL && sp != NULL);
  return Event::Msg(wxT("%s's %s reflects from %s's Magic Mirror back at %s."),
		    C_STR(p->name()), spell_list[ sp->id ].name, C_STR(b->name()), p->his());
}


Event msg_t_mirrored(Player* p, DoneSpell* sp) {
  wxASSERT(p != NULL && sp != NULL);
  return Event::Msg(wxT("%s's %s reflects from your Magic Mirror back at %s."),
		    C_STR(p->name()), spell_list[ sp->id ].name, p->his());
}


static const char* _monster_type(int type) {
  switch (type) {
  case CREATURE_GOBLIN:  return "Goblin";
  case CREATURE_OGRE:    return "Ogre";
  case CREATURE_TROLL:   return "Troll";
  case CREATURE_GIANT:   return "Giant";
  case CREATURE_FIRE_EL: return "Fire";
  case CREATURE_ICE_EL:  return "Ice";
  default: FAIL();
  }
  return NULL;
}


Event msg_o_summon_monsters(Player* p, int type, int howmany) {
  wxASSERT(p != NULL && howmany > 0);
  if (howmany == 1) {
    return Event::Msg(wxT("%s's %s appears by %s side."),
		      C_STR(p->name()), _monster_type(type), p->his());
  }
  return Event::Msg(wxT("%s's %s %ss appear by %s side."),
		    C_STR(p->name()), number(howmany), _monster_type(type), p->his());
}


Event msg_s_summon_monsters(int type, int howmany) {
  wxASSERT(howmany > 0);
  if (howmany == 1) {
    return Event::Msg(wxT("Your %s appears by your side."), _monster_type(type));
  }
  return Event::Msg(wxT("Your %s %ss appear by your side."), number(howmany), _monster_type(type));
}


Event msg_a_elementals_merge(int type, int howmany) {
  wxASSERT(howmany > 1);
  return Event::Msg(wxT("The %s %s Elementals merge into a single raging form."),
		    number(howmany), type == CREATURE_FIRE_EL ? "Fire" : "Ice");
}


Event msg_a_elemental_storm_1(int el_type, int storm_type) {
  return Event::Msg(wxT("The %s Elemental is dispersed into the %s Storm."),
		    el_type == CREATURE_FIRE_EL ? "Fire" : "Ice",
		    storm_type == SPELL_FIRE_STORM ? "Fire" : "Ice");
}


Event msg_a_elemental_storm_2(int el_type, int storm_type) {
  return Event::Msg(wxT("The %s Elemental is dispersed into the %s Storm, destroying both."),
		    el_type == CREATURE_FIRE_EL ? "Fire" : "Ice",
		    storm_type == SPELL_FIRE_STORM ? "Fire" : "Ice");
}


Event msg_a_elementals_kill_storm(int storm_type) {
  return Event::Msg(wxT("The Fire and Ice Elementals tear into each other, and whirl into a column of steam that dissipates in moments. The %s Storm is absorbed as well."),
		    storm_type == SPELL_FIRE_STORM ? "Fire" : "Ice");
}


Event msg_a_storms_kill_elemental(int el_type) {
  return Event::Msg(wxT("The Fire and Ice Storms tear each other into shreds of vapor, destroying the %s Elemental as well."),
		    el_type == SPELL_FIRE_STORM ? "Fire" : "Ice");
}


Event msg_o_fb_storm_cancel(Being* b) {
  wxASSERT(b != NULL);
  return Event::Msg(wxT("The Fireball hurtling towards %s is snuffed by the Ice Storm, leaving %s enveloped in a cloud of steam."),
		    C_STR(b->name()), b->him());
}


Event msg_o_perm_mind_cancel(Being* b, int spell, int perm_spell) {
  wxASSERT(b != NULL);
  return Event::Msg(wxT("The %s spell aimed at %s interferes with %s permanent %s and fizzles."),
		    spell_list[spell].name, C_STR(b->name()), b->his(),
		    spell_list[perm_spell].name);
}


Event msg_s_perm_mind_cancel(int spell, int perm_spell) {
  return Event::Msg(wxT("The %s spell aimed at you interferes with your permanent %s and fizzles, leaving you with a pounding headache."),
		    spell_list[spell].name, spell_list[perm_spell].name);
}


static wxString _make_spell_list(int spell[], int n_spells) {
  wxString list;
  for (int i = 0 ; i < n_spells ; i++) {
    list += spell_list[spell[i]].name;
    if (i != n_spells - 1) {
      list += wxT(", ");
    }
  }
  return list;
}

Event msg_o_mind_cancel(Being* b, int spell[], int n_spells) {
  wxASSERT(b != NULL && spell != NULL && n_spells > 0);
  return Event::Msg(wxT("The mind-control spells (%s) aimed at %s interfere with each other and fizzle."),
		    C_STR(_make_spell_list(spell, n_spells)), C_STR(b->name()));
}


Event msg_s_mind_cancel(int spell[], int n_spells) {
  wxASSERT(spell != NULL && n_spells > 0);
  return Event::Msg(wxT("The mind-control spells (%s) aimed at you interfere with each other and fizzle, leaving you with a pounding headache."),
		    C_STR(_make_spell_list(spell, n_spells)));
}


Event msg_o_hand_curl(Being* b, int spell, int hand) {
  wxASSERT(b != NULL && spell >= 0);
  return Event::Msg(wxT("The %s spell curls around %s's %s hand."),
		    spell_list[ spell ].name, C_STR(b->name()),
		    hand == LEFT_HAND ? "left" : "right");
}


Event msg_s_hand_curl(int spell, int hand) {
  wxASSERT(spell >= 0);
  return Event::Msg(wxT("The %s spell curls around your %s hand."),
		    spell_list[ spell ].name, hand == LEFT_HAND ? "left" : "right");
}


Event msg_a_charm_monster(Player* p, Being* b) {
  wxASSERT(p != NULL && b != NULL);
  return Event::Msg(wxT("The Charm Monster spell curls around %s, and %s smiles dopily at %s."),
		    C_STR(b->name()), b->he(), C_STR(p->name()));
}


Event msg_o_mind_afflicted(Being* b, int spell) {
  wxASSERT(b != NULL && spell >= 0);
  return Event::Msg(wxT("%s is still afflicted with %s."),
		    C_STR(b->name()), spell_list[ spell ].name);
}


Event msg_s_mind_afflicted(int spell) {
  wxASSERT(spell >= 0);
  return Event::Msg(wxT("You are still afflicted with %s."), spell_list[ spell ].name);
}


Event msg_s_paralyzed_still(int hand) {
  wxASSERT(hand < NUMBER_OF_HANDS);
  return Event::Msg(wxT("Your %s is still paralyzed."), which_hand(hand));
}


Event msg_o_paralyzed_still(Being* b, int hand) {
  wxASSERT(b != NULL);
  wxASSERT(hand < NUMBER_OF_HANDS);
  return Event::Msg(wxT("%s's %s is still paralyzed."), C_STR(b->name()), which_hand(hand));
}


Event msg_o_invis_fades(Being* b) {
  wxASSERT(b != NULL);
  return Event::Msg(wxT("A %s-shaped shadow appears, and %s rapidly becomes fully visible."),
		    C_STR(b->name()), b->he());
}


Event msg_o_dispel_stored(Player* p) {
  wxASSERT(p != NULL);
  return Event::Msg(wxT("The %s spell that %s had caught in a Delayed Effect is lost."),
		    spell_list[ p->banked() ].name, C_STR(p->name()));
}


Event msg_s_dispel_stored(Player* p) {
  wxASSERT(p != NULL);
  return Event::Msg(wxT("The %s spell that you had caught in a Delayed Effect is lost."),
		    spell_list[ p->banked() ].name);
}


Event msg_o_dispel_mind(Being* b, int spell) {
  wxASSERT(b != NULL);
  return Event::Msg(wxT("%s is freed of the %s."), C_STR(b->name()), spell_list[ spell ].name);
}


Event msg_s_dispel_mind(int spell) {
  return Event::Msg(wxT("You are freed of the %s."), spell_list[ spell ].name);
}


Event msg_o_stabs_miss(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The kni%s slide%s off of %s's Shield."),
		    howmany > 1 ? "ves" : "fe", howmany > 1 ? "" : "s", C_STR(b->name()));
}


Event msg_s_stabs_miss(int howmany) {
  wxASSERT(howmany > 0);
  return Event::Msg(wxT("The kni%s slide%s off of your Shield."),
		    howmany > 1 ? "ves" : "fe", howmany > 1 ? "" : "s");
}


Event msg_o_stabs_hit(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The kni%s strike%s %s."),
		    howmany > 1 ? "ves" : "fe", howmany > 1 ? "" : "s", C_STR(b->name()));
}


Event msg_s_stabs_hit(int howmany) {
  wxASSERT(howmany > 0);
  return Event::Msg(wxT("The kni%s strike%s you."),
		    howmany > 1 ? "ves" : "fe", howmany > 1 ? "" : "s");
}


Event msg_o_missiles_miss(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The Missile%s shatter%s on %s's Shield."),
		    howmany > 1 ? "s" : "", howmany > 1 ? "" : "s", C_STR(b->name()));
}


Event msg_s_missiles_miss(int howmany) {
  wxASSERT(howmany > 0);
  return Event::Msg(wxT("The Missile%s shatter%s on your Shield."),
		    howmany > 1 ? "s" : "", howmany > 1 ? "" : "s");
}


Event msg_o_missiles_hit(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The Missile%s strike%s %s."), howmany > 1 ? "s" : "",
		    howmany > 1 ? "" : "s", C_STR(b->name()));
}


Event msg_s_missiles_hit(int howmany) {
  wxASSERT(howmany > 0);
  return Event::Msg(wxT("The Missile%s strike%s you."),
		    howmany > 1 ? "s" : "", howmany > 1 ? "" : "s");
}


Event msg_o_cause_light_hit(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The%s Cause Light Wounds spell%s tear%s into %s."),
		    howmany > 1 ? space_number(howmany) : "", howmany > 1 ? "s" : "",
		    howmany > 1 ? "" : "s", C_STR(b->name()));
}


Event msg_s_cause_light_hit(int howmany) {
  wxASSERT(howmany > 0);
  return Event::Msg(wxT("The%s Cause Light Wounds spell%s tear%s into you."),
		    howmany > 1 ? space_number(howmany) : "", howmany > 1 ? "s" : "",
		    howmany > 1 ? "" : "s");
}


Event msg_o_cause_heavy_hit(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The%s Cause Heavy Wounds spell%s tear%s into %s."),
		    howmany > 1 ? space_number(howmany) : "", howmany > 1 ? "s" : "",
		    howmany > 1 ? "" : "s", C_STR(b->name()));
}


Event msg_s_cause_heavy_hit(int howmany) {
  wxASSERT(howmany > 0);
  return Event::Msg(wxT("The%s Cause Heavy Wounds spell%s tear%s painfully into you."),
		    howmany > 1 ? space_number(howmany) : "", howmany > 1 ? "s" : "",
		    howmany > 1 ? "" : "s");
}


Event msg_o_lightning_hit(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  if (howmany > 1) {
    return Event::Msg(wxT("The %s Lightning Bolts strike %s, staggering %s."),
		      number(howmany), C_STR(b->name()), b->him());
  }
  return Event::Msg(wxT("The Lightning Bolt staggers %s."), C_STR(b->name()));
}


Event msg_s_lightning_hit(int howmany) {
  wxASSERT(howmany > 0);
  if (howmany > 1) {
    return Event::Msg(wxT("The %s Lightning Bolts crash into you; the concussions nearly knock you off your feet."),
		      number(howmany));
  }
  return Event::Msg(wxT("The Lightning Bolt strikes you, staggering and nearly deafening you."));
}


Event msg_o_fireball_resist(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The%s Fireball%s strike%s %s, but the flames slide off %s Resist Heat aura."),
		    howmany > 1 ? space_number(howmany) : "",
		    howmany > 1 ? "s" : "", howmany > 1 ? "" : "s", C_STR(b->name()), b->his());
}


Event msg_s_fireball_resist(int howmany) {
  wxASSERT(howmany > 0);
  return Event::Msg(wxT("The%s Fireball%s strike%s you, but you feel only a slight warmth."),
		    howmany > 1 ? space_number(howmany) : "",
		    howmany > 1 ? "s" : "", howmany > 1 ? "" : "s");
}


Event msg_o_fireball_hit(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The%s Fireball%s engulf%s %s in flame."),
		    howmany > 1 ? space_number(howmany) : "", howmany > 1 ? "s" : "",
		    howmany > 1 ? "" : "s", C_STR(b->name()));
}


Event msg_s_fireball_hit(int howmany) {
  wxASSERT(howmany > 0);
  if (howmany > 1) {
    return Event::Msg(wxT("The %s Fireballs blast into you, surrounding you with blinding flame."),
		      number(howmany));
  }
  return Event::Msg(wxT("The Fireball strikes you with a roar of flame."));
}


Event msg_o_counter_storm(Being* b, int storm_type) {
  wxASSERT(b != NULL);
  return Event::Msg(wxT("The Counter-Spell around %s cancels the %s Storm near %s."),
		    C_STR(b->name()), storm_type == SPELL_FIRE_STORM ? "Fire" : "Ice", b->him());
}


Event msg_s_counter_storm(int storm_type) {
  return Event::Msg(wxT("The Counter-Spell cancels the %s Storm in your vicinity."),
		    storm_type == SPELL_FIRE_STORM ? "Fire" : "Ice");
}


Event msg_o_storm_hit(Being* b, int storm_type) {
  wxASSERT(b != NULL);
  return Event::Msg(wxT("The %s Storm washes over %s."),
		    storm_type == SPELL_FIRE_STORM ? "Fire" : "Ice", C_STR(b->name()));
}


Event msg_s_storm_hit(int storm_type) {
  return Event::Msg(wxT("The %s Storm washes over you."),
		    storm_type == SPELL_FIRE_STORM ? "Fire" : "Ice");
}


Event msg_o_fod_hit(Being* b) {
  wxASSERT(b != NULL);
  return Event::Msg(wxT("%s turns very pale and drops to %s knees, clutching at %s chest..."),
		    C_STR(b->name()), b->his(), b->his());
}


Event msg_o_cure_light(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The%s Cure Light Wounds spell%s weave%s around %s, and %s wounds fade."),
		    howmany > 1 ? space_number(howmany) : "",
		    howmany > 1 ? "s" : "", howmany > 1 ? "" : "s",
		    C_STR(b->name()), b->his());
}


Event msg_s_cure_light(int howmany) {
  wxASSERT(howmany > 0);
  return Event::Msg(wxT("The%s Cure Light Wounds spell%s weave%s around you, and the pain of your wounds lessens."),
		    howmany > 1 ? space_number(howmany) : "",
		    howmany > 1 ? "s" : "", howmany > 1 ? "" : "s");
}


Event msg_o_cure_heavy(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The%s Cure Heavy Wounds spell%s weave%s around %s, and %s wounds heal."),
		    howmany > 1 ? space_number(howmany) : "",
		    howmany > 1 ? "s" : "", howmany > 1 ? "" : "s",
		    C_STR(b->name()), b->his());
}


Event msg_s_cure_heavy(int howmany) {
  wxASSERT(howmany > 0);
  return Event::Msg(wxT("The%s Cure Heavy Wounds spell%s weave%s around you, and your wounds heal considerably."),
		    howmany > 1 ? space_number(howmany) : "",
		    howmany > 1 ? "s" : "", howmany > 1 ? "" : "s");
}


Event msg_o_raise_alive(Being* b, int howmany) {
  wxASSERT(b != NULL && howmany > 0);
  return Event::Msg(wxT("The%s Raise Dead spell%s burn%s through %s. Although %s is not dead, the spell%s have a powerful healing influence."),
		    howmany > 1 ? space_number(howmany) : "",
		    howmany > 1 ? "s" : "", howmany > 1 ? "" : "s",
		    C_STR(b->name()), b->he(), howmany > 1 ? "s" : "");
}


Event msg_s_raise_alive(int howmany) {
  wxASSERT(howmany > 0);
  return Event::Msg(wxT("The%s Raise Dead spell%s burn%s through you. Although you aren't dead, the spell%s a powerful healing influence."),
		    howmany > 1 ? space_number(howmany) : "", howmany > 1 ? "s" : "",
		    howmany > 1 ? "" : "s", howmany > 1 ? "s have" : " has");
}


Event msg_o_raise_dead(Being* b) {
  wxASSERT(b != NULL);
  return Event::Msg(wxT("%s twitches, glows, and then jumps to %s feet! Except for a bit of incipient mold, %s is in perfect health again."),
		    C_STR(b->name()), b->his(), b->he());
}
