// MemBuf.h: Because wxMemoryBuffer is so completely inadequate.

#include "Util.h"


MemBuf::MemBuf(size_t bytes) : wxMemoryBuffer(bytes) {}


MemBuf::MemBuf(const wxString& str) {
  append(str);
}


wxUint8 MemBuf::operator[](size_t index) {
  REQUIRE(index < length());
  return data()[index];
}


void MemBuf::trim_start(size_t bytes) {
  REQUIRE(bytes <= length());
  memmove(data(), data() + bytes, length());
  SetDataLen(length() - bytes);
}


void MemBuf::trim_end(size_t bytes) {
  REQUIRE(bytes <= length());
  SetDataLen(length() - bytes);
}


void MemBuf::prepend(const wxUint8* input, size_t bytes) {
  wxASSERT(input != NULL);
  wxUint8* buf = (wxUint8*) GetWriteBuf(length() + bytes);
  memmove(buf + bytes, buf, length());
  memmove(buf, input, bytes);
  UngetWriteBuf(length() + bytes);
}


void MemBuf::append(const wxUint8* input, size_t bytes) {
  wxASSERT(input != NULL);
  AppendData(input, bytes);
}


void MemBuf::append(const wxString& str) {
  append((const wxUint8*) C_STR(str), str.length());
}


void MemBuf::append(const MemBuf& buf) {
  append(buf.data(), buf.length());
}


void MemBuf::append(unsigned char ch) {
  append((const wxUint8*) &ch, 1);
}


wxString MemBuf::as_str(size_t start, size_t len) const {
  REQUIRE(start <= length());
  REQUIRE(len == MemBuf::ALL || start + len <= length());
  if (len == MemBuf::ALL) {
    return wxString((const char*) data() + start, wxConvUTF8, length() - start);
  } else {
    return wxString((const char*) data() + start, wxConvUTF8, len);
  }
}


MemBuf MemBuf::subset(size_t start, size_t len) const {
  REQUIRE(start <= length());
  REQUIRE(len == MemBuf::ALL || start + len <= length());
  wxUint32 size = (len == MemBuf::ALL ? length() - start : len);
  MemBuf mb(size);
  mb.AppendData(data() + start, size);
  return mb;
}


void MemBuf::dump() const {
  fprintf(stderr, "MemBuf 0x%p (%d bytes):", this, length());
  for (size_t i = 0 ; i < length() ; i++) {
    if (i % 32 == 0) fputs("\n    ", stderr);
    else if (i % 4 == 0) fputc(' ', stderr);
    fprintf(stderr, "%02x", data()[i]);
  }
  fputc('\n', stderr);
}
