/* $Id: Images.cpp,v 1.2 2008/04/12 23:34:11 dpt Exp $
 *
 * Images.cpp: Loading and storing bitmap images.
 */

#include "Graphics.h"
#include "Spellcast.h"
#include <wx/xpmdecod.h>


struct image {
  int id;
  const char** xpm;
  int flip;
};


struct image imagelist[ NUMBER_OF_IMAGES ] = {
  { NO_GESTURE,             nothing_xpm,      0 },
  { ANTISPELL_GESTURE,      antispell_xpm,    0 },
  { UNKNOWN_GESTURE,        unknown_xpm,      0 },
  { BIG_HAND_LEFT_CLAP,     left_clap_xpm,    0 },
  { BIG_HAND_LEFT_DIGIT,    left_digit_xpm,   0 },
  { BIG_HAND_LEFT_FINGERS,  left_fingers_xpm, 0 },
  { BIG_HAND_LEFT_KNIFE,    left_knife_xpm,   0 },
  { BIG_HAND_LEFT_PALM,     left_palm_xpm,    0 },
  { BIG_HAND_LEFT_SNAP,     left_snap_xpm,    0 },
  { BIG_HAND_LEFT_WAVE,     left_wave_xpm,    0 },
  { BIG_HAND_RIGHT_CLAP,    left_clap_xpm,    1 },
  { BIG_HAND_RIGHT_DIGIT,   left_digit_xpm,   1 },
  { BIG_HAND_RIGHT_FINGERS, left_fingers_xpm, 1 },
  { BIG_HAND_RIGHT_KNIFE,   left_knife_xpm,   1 },
  { BIG_HAND_RIGHT_PALM,    left_palm_xpm,    1 },
  { BIG_HAND_RIGHT_SNAP,    left_snap_xpm,    1 },
  { BIG_HAND_RIGHT_WAVE,    left_wave_xpm,    1 },
  { SMALL_CLAP,             small_clap_xpm,      0 },
  { SMALL_DIGIT,            small_digit_xpm,     0 },
  { SMALL_DOUBLE_DIGIT,     small_2_digits_xpm,  0 },
  { SMALL_FINGERS,          small_fingers_xpm,   0 },
  { SMALL_DOUBLE_FINGERS,   small_2_fingers_xpm, 0 },
  { SMALL_PALM,             small_palm_xpm,      0 },
  { SMALL_DOUBLE_PALM,      small_2_palms_xpm,   0 },
  { SMALL_SNAP,             small_snap_xpm,      0 },
  { SMALL_DOUBLE_SNAP,      small_2_snaps_xpm,   0 },
  { SMALL_WAVE,             small_wave_xpm,      0 },
  { SMALL_DOUBLE_WAVE,      small_2_waves_xpm,   0 },
  { SPELL_LIST,             spelllist_xpm,       0 }
};


// The following three functions assume that the XPM is a large 48x48
// gesture icon, because I'm lazy and that's all the functionality I
// need from them anyhow.

static char** _flip_xpm(char** xpm) {
  REQUIRE(strcmp(xpm[0], "48 48 2 1") == 0);

  for (int i = 3 ; i < 48 + 3 ; i++) {
    int len = strlen(xpm[i]);
    for (int x = 0 ; x < len / 2 ; x++) {
      char tmp = xpm[i][ len - 1 - x ];
      xpm[i][ len - 1 - x ] = xpm[i][x];
      xpm[i][x] = tmp;
    }
  }
  return xpm;
}


static char** _copy_xpm(const char** xpm) {
  REQUIRE(strcmp(xpm[0], "48 48 2 1") == 0);

  char** newxpm = new char*[ 48 + 3 ];
  for (int i = 0 ; i < 48 + 3 ; i++) {
    newxpm[i] = strdup(xpm[i]);
  }
  return newxpm;
}


static void _free_xpm(char** xpm) {
  REQUIRE(strcmp(xpm[0], "48 48 2 1") == 0);

  for (int i = 0 ; i < 48 + 3 ; i++) {
    free(xpm[i]);
  }
  delete [] xpm;
}


Images::Images() {
  bitmaps = new wxBitmap[ NUMBER_OF_IMAGES ];

  for (int i = 0 ; i < NUMBER_OF_IMAGES ; i++) {
    if (imagelist[i].flip) {
      char** xpm = _copy_xpm(imagelist[i].xpm);
      bitmaps[i] = wxBitmap(_flip_xpm(xpm));
      _free_xpm(xpm);
    } else {
      bitmaps[i] = wxBitmap(imagelist[i].xpm);
    }
  }
}


Images::~Images() {
  delete [] bitmaps;
}


wxBitmap* Images::get_image(int which) const {
  wxASSERT(which >= 0);
  wxASSERT(which < NUMBER_OF_IMAGES);
  return &bitmaps[ which ];
}


wxBitmap Images::get_gesture_image(int hand, int gesture) const {
  wxASSERT(hand < NUMBER_OF_HANDS);
  wxASSERT(gesture >= 0);
  wxASSERT(gesture < NUMBER_OF_GESTURES);

  int image_id = NO_GESTURE;
  switch (hand) {
  case LEFT_HAND:
    switch (gesture) {
      case GESTURE_ANTISPELL: image_id = ANTISPELL_GESTURE;     break;
      case GESTURE_CLAP:      image_id = BIG_HAND_LEFT_CLAP;    break;
      case GESTURE_DIGIT:     image_id = BIG_HAND_LEFT_DIGIT;   break;
      case GESTURE_FINGERS:   image_id = BIG_HAND_LEFT_FINGERS; break;
      case GESTURE_KNIFE:     image_id = BIG_HAND_LEFT_KNIFE;   break;
      case GESTURE_NOTHING:   image_id = NO_GESTURE;            break;
      case GESTURE_PALM:      image_id = BIG_HAND_LEFT_PALM;    break;
      case GESTURE_SNAP:      image_id = BIG_HAND_LEFT_SNAP;    break;
      case GESTURE_UNKNOWN:   image_id = UNKNOWN_GESTURE;       break;
      case GESTURE_WAVE:      image_id = BIG_HAND_LEFT_WAVE;    break;
      default:	FAIL();
    }
    break;

  case RIGHT_HAND:
    switch (gesture) {
      case GESTURE_ANTISPELL: image_id = ANTISPELL_GESTURE;      break;
      case GESTURE_CLAP:      image_id = BIG_HAND_RIGHT_CLAP;    break;
      case GESTURE_DIGIT:     image_id = BIG_HAND_RIGHT_DIGIT;   break;
      case GESTURE_FINGERS:   image_id = BIG_HAND_RIGHT_FINGERS; break;
      case GESTURE_KNIFE:     image_id = BIG_HAND_RIGHT_KNIFE;   break;
      case GESTURE_NOTHING:   image_id = NO_GESTURE;             break;
      case GESTURE_PALM:      image_id = BIG_HAND_RIGHT_PALM;    break;
      case GESTURE_SNAP:      image_id = BIG_HAND_RIGHT_SNAP;    break;
      case GESTURE_UNKNOWN:   image_id = UNKNOWN_GESTURE;        break;
      case GESTURE_WAVE:      image_id = BIG_HAND_RIGHT_WAVE;    break;
      default:	FAIL();
    }
    break;

  default:  FAIL();
  }

  return *get_image(image_id);
}


wxIcon Images::get_icon() const {
  wxIcon icon = wxICON(left_snap);
  wxASSERT(icon.Ok());
  return icon;
}

