#include "Spellcast.h"
#include <wx/listimpl.cpp>
WX_DEFINE_LIST(QueryList);


void GameState::_clear_everything() {
  unsigned int i;

  for (i = 0 ; i < MAX_PLAYERS ; i++) {
    if (m_players[i] != NULL) {
      delete player(i);
      m_players[i] = NULL;
    }
  }
  if (m_creatures) {
    for (i = 0 ; i < m_max_creatures ; i++) {
      if (m_creatures[i] != NULL) {
	delete creature(i);
	m_creatures[i] = NULL;
      }
    }
  }

  m_player_count = m_creature_count = m_max_players = m_server_port = m_turn = 0;
  m_server_name = wxT("");
  m_game_state = NO_GAME_YET;
  m_phase = NO_PHASE;
}


GameState::GameState() {
  unsigned int i;

  m_game_state = GAME_NOT_STARTED;
  m_max_creatures = DEFAULT_MAX_CREATURES;
  m_creatures = new Creature*[ DEFAULT_MAX_CREATURES ];

  for (i = 0 ; i < MAX_PLAYERS ; i++) {
    m_players[i] = NULL;
  }
  for (i = 0 ; i < m_max_creatures ; i++) {
    m_creatures[i] = NULL;
  }
  _clear_everything();
}


GameState::GameState(MemBuf* data, wxInt8 me) {
  wxUint32 i, x;
  wxUint8 n;

  m_game_state = GAME_NOT_STARTED;
  m_creatures = NULL;

  for (i = 0 ; i < MAX_PLAYERS ; i++) {
    m_players[i] = NULL;
  }
  _clear_everything();

  if (unserialize(data, "BBIIBBTSI", &m_player_count, &m_max_players, &m_creature_count,
		   &m_max_creatures, &m_game_state, &m_phase, &m_server_name, &m_server_port,
		   &m_turn) == 0
      || m_max_players > MAX_PLAYERS || m_player_count > m_max_players) {
    FAIL_MSG("Bogus GameState serialize string!");
  }

  m_creatures = new Creature*[ m_max_creatures ];
  for (i = 0 ; i < m_max_creatures ; i++) {
    m_creatures[i] = NULL;
  }

  for (i = 0 ; i < m_player_count ; i++) {
    if (unserialize(data, "B", &n) == 0 || n >= MAX_PLAYERS) {
      FAIL_MSG("Bogus GameState player string!");
    }
    m_players[n] = new Player(data);
    if (m_turn == 0) {
      m_players[n]->clear_gestures();
    }
  }

  for (i = 0 ; i < m_creature_count ; i++) {
    if (unserialize(data, "I", &x) == 0 || x > m_max_creatures) {
      FAIL_MSG("Bogus GameState creature string!");
    }
    m_creatures[x] = new Creature(data);
  }

  wxASSERT(data->length() == 0);

  if (me >= 0) {
    wxASSERT(me < (int) MAX_PLAYERS);
    wxASSERT(player(me) != NULL);
    player(me)->this_is_me();
  }
  m_game_state = GAME_NOT_STARTED;
}


// It has occurred to me that I never use these functions, so I'll leave
// suicide stubs just in case.
GameState::GameState(const GameState& WXUNUSED(other)) {
  FAIL();
}


GameState& GameState::operator=(const GameState& WXUNUSED(f)) {
  FAIL();
  return *this;
}


GameState::~GameState() {
  _clear_everything();
  delete [] m_creatures;
}


MemBuf GameState::serialize() const {
  wxUint32 i;
  wxUint8 byte;
  MemBuf foo = ::serialize("BBIIBBTSI", m_player_count, m_max_players, m_creature_count,
			  m_max_creatures, m_game_state, m_phase, &m_server_name, m_server_port,
			  m_turn);

  for (i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL) {
      byte = i;
      foo.append(::serialize("B", byte));
      foo.append(player(i)->serialize());
    }
  }
  for (i = 0 ; i < m_max_creatures ; i++) {
    if (creature(i) != NULL) {
      foo.append(::serialize("I", i));
      foo.append(creature(i)->serialize());
    }
  }
  return foo;
}


Player* GameState::me() const {
  unsigned int i;

  for (i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL && player(i)->is_me()) {
      return player(i);
    }
  }
  return NULL;
}


Being* GameState::being(wxUint32 i) const {
  wxASSERT(i < MAX_PLAYERS + m_max_creatures);
  if (i < MAX_PLAYERS) {
    return m_players[i];
  }
  return m_creatures[i - MAX_PLAYERS];
}


void GameState::add_player(wxUint8 id, const wxString& name, wxUint8 gender) {
  add_player(id, new Player(name, gender));
}


void GameState::add_player(wxUint8 id, Player* player) {
  wxASSERT(player->name().Length() > 0);
  wxASSERT(id < MAX_PLAYERS);
  wxASSERT(m_players[id] == NULL);

  m_players[id] = player;
  m_player_count++;
}


void GameState::replace_player(wxUint8 id, Player* p) {
  wxASSERT(p->name().Length() > 0);
  wxASSERT(id < MAX_PLAYERS);
  wxASSERT(player(id) != NULL);

  for (int i = MAX_GESTURES - 1 ; i >= 0 ; i--) {
    player(id)->add_gestures(Event::SvHands(id, p->get_gesture(LEFT_HAND, i),
					    p->get_gesture(RIGHT_HAND, i)));
  }
}


void GameState::remove_player(wxUint8 id) {
  wxASSERT(id < MAX_PLAYERS);
  wxASSERT(m_players[id] != NULL);

  delete player(id);
  m_players[id] = NULL;
  m_player_count--;
  if (m_game_state == GAME_IN_PROGRESS) {
    for (unsigned int i = 0 ; i < m_max_creatures ; i++) {
      if (creature(i) != NULL && creature(i)->owner() == id) {
	creature(i)->set_owner(-1);
      }
    }
  }
}


void GameState::add_creature(Creature* thing) {
  wxUint32 i;
  wxASSERT(thing != NULL);

  for (i = 0 ; i < m_max_creatures ; i++) {
    if (m_creatures[i] == NULL) {
      m_creatures[i] = thing;
      m_creature_count++;
      return;
    }
  }

  wxUint32 n = m_max_creatures * 2;
  Creature** foo = new Creature*[ n ];
  for (i = 0 ; i < n ; i++) {
    foo[i] = (i < m_max_creatures) ? m_creatures[i] : NULL;
  }
  delete [] m_creatures;
  m_creatures = foo;
  wxASSERT(m_creatures[ m_max_creatures ] == NULL);
  m_creatures[ m_max_creatures ] = thing;
  m_creature_count++;
  m_max_creatures = n;
}


void GameState::remove_creature(wxUint32 id) {
  wxASSERT(g_server != NULL);
  wxASSERT(id < m_max_creatures);
  wxASSERT(m_creatures[id] != NULL);

  delete m_creatures[id];
  m_creatures[id] = NULL;
  m_creature_count--;
}


void GameState::set_max_players(wxUint8 max_players) {
  wxASSERT(max_players <= MAX_PLAYERS);
  m_max_players = max_players;
}


void GameState::set_game_state(wxUint8 new_state) {
  wxASSERT(new_state < NUMBER_OF_GAME_STATES);
  m_game_state = new_state;
}


void GameState::set_server_name(const wxString& name) {
  wxASSERT(name.Len() > 0);
  m_server_name = wxString(name);
}


void GameState::set_server_port(wxUint16 port) {
  m_server_port = port;
}


void GameState::set_done_players(wxUint32 vec) {
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL) {
      player(i)->set_done((vec & (1 << i)) != 0);
    }
  }
}


bool GameState::name_is_taken(const wxString& name) const {
  if (name == wxT("(nobody)") || name == wxT("nobody in particular")) {
    return true;
  }
  for (unsigned int i = 0 ; i < MAX_PLAYERS ; i++) {
    if (player(i) != NULL && player(i)->name() == name) {
      return true;
    }
  }
  return false;
}


PlayerIterator::PlayerIterator(const GameState* state) {
  wxASSERT(state != NULL);
  wxASSERT(state->player(0) != NULL);

  m_state = (GameState*) state;
  m_pos = 0;
}


Player* PlayerIterator::operator*() const {
  if (m_pos < m_state->max_players()) {
    wxASSERT(m_state->player(m_pos) != NULL);
    return m_state->player(m_pos);
  }
  return NULL;
}


const PlayerIterator& PlayerIterator::operator++() {
  do { m_pos++; } while (m_pos < m_state->max_players() && m_state->player(m_pos) == NULL);
  return *this;
}


PlayerIterator PlayerIterator::operator++(int WXUNUSED(postfix)) {
  PlayerIterator foo = *this;
  do { m_pos++; } while (m_pos < m_state->max_players() && m_state->player(m_pos) == NULL);
  return foo;
}


bool PlayerIterator::done() const {
  return m_pos >= MAX_PLAYERS;
}


CreatureIterator::CreatureIterator(const GameState* state) {
  wxASSERT(state != NULL);
  m_state = (GameState*) state;
  m_pos = 0;

  while (m_pos < m_state->creature_array_size() && m_state->creature(m_pos) == NULL) {
    m_pos++;
  }
}


Creature* CreatureIterator::operator*() const {
  if (m_pos < m_state->creature_array_size()) {
    wxASSERT(m_state->creature(m_pos) != NULL);
    return m_state->creature(m_pos);
  }
  return NULL;
}


const CreatureIterator& CreatureIterator::operator++() {
  do { m_pos++; } while (m_pos < m_state->creature_array_size() &&
			 m_state->creature(m_pos) == NULL);
  return *this;
}


CreatureIterator CreatureIterator::operator++(int WXUNUSED(postfix)) {
  CreatureIterator foo = *this;
  do { m_pos++; } while (m_pos < m_state->creature_array_size() &&
			 m_state->creature(m_pos) == NULL);
  return foo;
}


bool CreatureIterator::done() const {
  return m_pos >= m_state->creature_array_size();
}


BeingIterator::BeingIterator(const GameState* state) {
  wxASSERT(state != NULL);
  wxASSERT(state->being(0) != NULL);

  m_state = (GameState*) state;
  m_pos = 0;
}


Being* BeingIterator::operator*() const {
  if (m_pos < m_state->max_players() + m_state->creature_array_size()) {
    wxASSERT(m_state->being(m_pos) != NULL);
    return m_state->being(m_pos);
  }
  return NULL;
}


const BeingIterator& BeingIterator::operator++() {
  do {
    m_pos++;
  } while (m_pos < m_state->max_players() + m_state->creature_array_size() &&
	   m_state->being(m_pos) == NULL);
  return *this;
}


BeingIterator BeingIterator::operator++(int WXUNUSED(postfix)) {
  BeingIterator foo = *this;
  do {
    m_pos++;
  } while (m_pos < m_state->max_players() + m_state->creature_array_size() &&
	   m_state->being(m_pos) == NULL);
  return foo;
}


bool BeingIterator::done() const {
  return m_pos >= MAX_PLAYERS + m_state->creature_array_size();
}

