/* $Id: Frames.cpp,v 1.21 2008/05/04 09:19:51 dpt Exp $
 *
 * Frames.h: Code for several custom frames that we have to pop up.
 */

#include "Spellcast.h"


const int PLAYER_WIDTH  = 100;
const int PLAYER_HEIGHT = 425 + 20;   // add a little slop for the scrollbar
const int MIN_IN_BETWEEN = 15;
const int MIN_WINSIZE = 4;
const int MIN_WIDTH = MIN_WINSIZE * PLAYER_WIDTH + (MIN_WINSIZE - 1) * MIN_IN_BETWEEN;
const int SCROLL_UNITS = 10;
const int GESTURE_WIDTH = 48;
const int GESTURE_HEIGHT = 48;
const int ARENA_HEIGHT = 6;


static wxRect positions[8]; 
struct _pt { int x, y; };
static struct _pt position_data[8] = {
  { 2 + ((GESTURE_WIDTH + 2) * 0),  2 },
  { 2 + ((GESTURE_WIDTH + 2) * 1),  2 },
  { 2 + ((GESTURE_WIDTH + 2) * 2),  2 },
  { 2 + ((GESTURE_WIDTH + 2) * 3),  2 },
  { 2 + ((GESTURE_WIDTH + 2) * 0), 4 + GESTURE_HEIGHT },
  { 2 + ((GESTURE_WIDTH + 2) * 1), 4 + GESTURE_HEIGHT },
  { 2 + ((GESTURE_WIDTH + 2) * 2), 4 + GESTURE_HEIGHT },
  { 2 + ((GESTURE_WIDTH + 2) * 3), 4 + GESTURE_HEIGHT },
};


wxPoint PickWindow::_get_position(wxPoint pos, int gesture) {
  return wxPoint(pos.x - position_data[gesture].x, pos.y - position_data[gesture].y);
}


PickWindow::PickWindow(int hand, int current_gesture, wxPoint pos) :
  wxFrame(g_window, -1, wxT("Pick Gesture"), _get_position(pos, current_gesture),
	  wxDefaultSize, wxSIMPLE_BORDER | wxFRAME_NO_TASKBAR | wxFRAME_FLOAT_ON_PARENT)
{
  m_grey    = new wxPen(*wxLIGHT_GREY, 2, wxSOLID);
  m_black   = new wxPen(*wxBLACK, 2, wxSOLID);
  m_hand    = hand;
  m_gesture = current_gesture;

  SetClientSize(GESTURE_WIDTH * 4 + 10, GESTURE_HEIGHT * 2 + 6);
  Raise();
  Show();

  for (int i = 0 ; i < 8 ; i++) {
    positions[i] = wxRect(ClientToScreen(wxPoint(position_data[i].x, position_data[i].y)),
			   wxSize(GESTURE_WIDTH, GESTURE_HEIGHT));
  }
}


PickWindow::~PickWindow() {
  delete m_grey;
  delete m_black;
}


int PickWindow::get_gesture(wxPoint screen_coords) {
  for (int i = 0 ; i < 8 ; i++) {
    if (positions[i].Contains(screen_coords)) {
      return i;
    }
  }
  return -1;
}


void PickWindow::OnPaint(wxPaintEvent& WXUNUSED(event)) {
  wxPaintDC dc(this);
  dc.SetBackgroundMode(wxTRANSPARENT);
  dc.SetBackground(*wxLIGHT_GREY_BRUSH);
  dc.Clear();

  for (int i = 0 ; i < 8 ; i++) {
    if (positions[i].Contains(wxGetMousePosition())) {
      dc.SetBrush(*wxBLACK_BRUSH);
      dc.SetPen(*wxBLACK_PEN);
      dc.DrawRectangle(position_data[i].x - 2, position_data[i].y - 2,
			GESTURE_WIDTH + 4, GESTURE_HEIGHT + 4);
    }
    dc.DrawBitmap(g_images->get_gesture_image(m_hand, i),
		   position_data[i].x, position_data[i].y, false);
  }
}

void PickWindow::OnDrag(wxPoint location) {
  int cur = get_gesture(location);
  if (cur != m_gesture) {
    wxClientDC dc(this);
    int width = GetSize().GetWidth();
    int height = GetSize().GetHeight();

    dc.SetPen(*m_grey);
    dc.DrawLine(0, 1, width, 1);
    dc.DrawLine(2, 51, width - 4, 51);
    dc.DrawLine(0, 101, width, 101);
    dc.DrawLine(1, 2, 1, height - 4);
    dc.DrawLine(51, 2, 51, height - 4);
    dc.DrawLine(101, 2, 101, height - 4);
    dc.DrawLine(151, 2, 151, height - 4);
    dc.DrawLine(201, 2, 201, height - 4);
    dc.SetPen(*m_black);
    int x = position_data[cur].x - 1;
    int y = position_data[cur].y - 1;
    dc.DrawLine(x, y, x + GESTURE_WIDTH + 2, y);
    dc.DrawLine(x + GESTURE_WIDTH + 2, y, x + GESTURE_WIDTH + 2, y + GESTURE_HEIGHT + 2);
    dc.DrawLine(x, y, x, y + GESTURE_HEIGHT + 2);
    dc.DrawLine(x, y + GESTURE_HEIGHT + 2, x + GESTURE_WIDTH + 2, y + GESTURE_HEIGHT + 2);
    m_gesture = cur;
  }
}


LogWindow::LogWindow(wxWindow* parent, wxTextCtrl* input, const wxPoint& pos, const wxSize& size)
  : wxTextCtrl(parent, -1, wxT("Welcome to " PACKAGE "!\n"), pos, size,
	       wxTE_RICH | wxTE_MULTILINE | wxTE_READONLY)
{ 
  m_input = input;
}


void LogWindow::OnKeyboard(wxKeyEvent& event) {
  // FIXME: send event to m_input control (ProcessEvent and AddPendingEvent don't work)
  m_input->SetFocus();
}


void LogWindow::AppendText(const wxString& text) {
  wxTextCtrl::AppendText(text);
  SetScrollPos(wxVERTICAL, GetScrollRange(wxVERTICAL), true);
}


GestureWindow::GestureWindow(wxWindow* parent) : 
  wxScrolledWindow(parent, -1, wxDefaultPosition, wxDefaultSize, wxHSCROLL)
{
  SetScrollbars(1, 0, MIN_WIDTH, 0);
  SetMinSize(wxSize(MIN_WIDTH, PLAYER_HEIGHT));
  m_picker = NULL;
  SetClickable(true);
}


void GestureWindow::OnDraw(wxDC& dc) {
  dc.SetBackgroundMode(wxTRANSPARENT);

  if (g_client == NULL || g_client->state().game_state() == NO_GAME_YET) {
    int width, height;
    wxSize sz = GetSize();
    dc.GetTextExtent(wxT("Not connected."), &width, &height);
    dc.DrawText(wxT("Not connected."), sz.GetWidth() / 2 - width / 2,
		sz.GetHeight() / 2 - height / 2);

  } else {
    Player* players[ MAX_PLAYERS ];
    wxUint8 columns = 0;

    for (unsigned int i = 0 ; i < g_client->state().max_players() ; i++) {
      if (g_client->state().player(i) != NULL) {
	players[ columns++ ] = g_client->state().player(i);
      }
    }
    SetVirtualSize(PLAYER_WIDTH * columns + MIN_IN_BETWEEN * columns, PLAYER_HEIGHT);

    for (unsigned int i = 0 ; i < columns ; i++) {
      int width = GetClientSize().GetWidth();
      int w = PLAYER_WIDTH * i + MIN_IN_BETWEEN * i;
      int max = PLAYER_WIDTH * columns + MIN_IN_BETWEEN * (columns - 1);
      if (max >= width) {
	_DrawPlayer(dc, players[i], wxPoint(w, 0));
      } else {
	_DrawPlayer(dc, players[i], wxPoint(w + (width - max) / 2, 0));
      }
    }
  }
}


void GestureWindow::_DrawPlayer(wxDC& dc, const Player* player, wxPoint where) {
  wxASSERT(player != NULL);
  wxASSERT(where.y == 0);

  int width, height;
  dc.GetTextExtent(player->name(), &width, &height);
  dc.DrawText(player->name().c_str(), where.x, where.y);

  int down = where.y + height + 2;
  for (int y = 0 ; y < MAX_GESTURES ; y++) {
    int left_id  = player->get_gesture(LEFT_HAND, (MAX_GESTURES - y - 1));
    int right_id = player->get_gesture(RIGHT_HAND, (MAX_GESTURES - y - 1));
    wxBitmap lefthand  = g_images->get_gesture_image(LEFT_HAND, left_id);
    wxBitmap righthand = g_images->get_gesture_image(RIGHT_HAND, right_id);
    dc.DrawBitmap(lefthand, where.x, down, false);
    dc.DrawBitmap(righthand, where.x + GESTURE_WIDTH + 2, down, false);
    down += GESTURE_HEIGHT + 2;
  }

  if (player->is_me()) {
    m_boxes[ LEFT_HAND ] = wxRect(where.x, down - GESTURE_HEIGHT - 2,
				  GESTURE_WIDTH, GESTURE_HEIGHT);
    m_boxes[ RIGHT_HAND ] = wxRect(where.x + GESTURE_WIDTH + 2, down - GESTURE_HEIGHT - 2,
				    GESTURE_WIDTH, GESTURE_HEIGHT);
  }
}


int GestureWindow::_clicked_on_hand(wxPoint pt) {
  if (m_boxes[ LEFT_HAND ].Contains(pt)) {
    return LEFT_HAND;
  } else if (m_boxes[ RIGHT_HAND ].Contains(pt)) {
    return RIGHT_HAND;
  }
  return NEITHER_HAND;
}


void GestureWindow::OnMouse(wxMouseEvent& event) {
  int hand, newx, newy;
  wxPoint pos = event.GetPosition();

  CalcUnscrolledPosition(pos.x, pos.y, &newx, &newy);
  if (event.ButtonDown() && m_picker == NULL && m_click == true &&
      ((hand = _clicked_on_hand(wxPoint(newx, newy))) != NEITHER_HAND)) {
    // && g_client->state().game_state() == GAME_IN_PROGRESS) {
    CaptureMouse();
    CalcScrolledPosition(m_boxes[hand].x, m_boxes[hand].y, &newx, &newy);
    m_picker = new PickWindow(hand, g_client->state().me()->get_gesture(hand, 0),
			      ClientToScreen(wxPoint(newx, newy)));

  } else if (event.Dragging() && m_picker != NULL) {
    m_picker->OnDrag(ClientToScreen(pos));

  } else if (event.ButtonUp() && m_picker != NULL) {
    int gesture = m_picker->get_gesture(ClientToScreen(pos));
    if (gesture >= 0) {
      // only one knife allowed at a time!
      if (gesture == GESTURE_KNIFE && g_client->state().me()->get_gesture(m_picker->hand() == LEFT_HAND ? RIGHT_HAND : LEFT_HAND, 0) == GESTURE_KNIFE) {
	gesture = GESTURE_NOTHING;
      }
      g_client->state().me()->set_gesture(m_picker->hand(), gesture);
    }
    ReleaseMouse();
    delete m_picker;
    m_picker = NULL;
    Refresh();
  }
}


SpellWindow::SpellWindow(wxBitmap* img) : wxFrame(g_window, -1, wxT("Spell List"), wxDefaultPosition, wxDefaultSize, wxDEFAULT_FRAME_STYLE & ~(wxRESIZE_BORDER | wxMAXIMIZE_BOX)) {
  wxASSERT(img != NULL);
  m_bitmap = img;
  SetClientSize(img->GetWidth(), img->GetHeight());
  CenterOnParent();
  Show();
}


SpellWindow::~SpellWindow() {
  g_window->OnSpellListDismiss();
}


void SpellWindow::OnPaint(wxPaintEvent& WXUNUSED(event)) {
  wxPaintDC dc(this);
  dc.DrawBitmap(*m_bitmap, 0, 0, 0);
}


ArenaList::ArenaList(wxWindow* parent, int id) :
  wxListBox(parent, id, wxDefaultPosition, wxDefaultSize, 0, NULL, wxLB_SINGLE | wxLB_ALWAYS_SB) {
  wxASSERT(parent != NULL);
  // monospaced font -- looks ugly, but alignment is correct
  SetFont(wxFont(GetFont().GetPointSize(), wxMODERN, wxNORMAL, wxNORMAL));

  wxClientDC dc(this);
  wxCoord w, h;
  wxFont font = GetFont();
  GetTextExtent(wxT("                                                 "), &w, &h, NULL, NULL, &font);
  SetMinSize(wxSize(w, h * (ARENA_HEIGHT + 2)));
}


void ArenaList::UpdateList() {
  Clear();
}


void ArenaList::UpdateList(const GameState& state) {
  unsigned int i;

  Clear();
  for (i = 0 ; i < MAX_PLAYERS ; i++) {
    Player* p = state.player(i);
    if (p && p->is_alive()) {
      Append(wxString::Format(wxT(" %c %-32s  %2d  %s"), (!p->active() || p->done()) ? ' ' : '*',
			      p->name().c_str(), p->health(), p->status().c_str()));

      for (CreatureIterator it = state.creatures_iter() ; !it.done() ; it++) {
	if ((*it)->owner() == (int) i && (*it)->is_alive()) {
	  Append(wxString::Format(wxT("      %-29s  %2d  %s"), (*it)->name().c_str(),
				  (*it)->health(), (*it)->status().c_str()));
	}
      }
    }
  }

  for (CreatureIterator it = state.creatures_iter() ; !it.done() ; it++) {
    if ((*it)->is_alive() && (*it)->owner() < 0) {
      Append(wxString::Format(wxT("   %-32s  %2d  %s"), (*it)->name().c_str(),
				(*it)->health(), (*it)->status().c_str()));
    }
  }
}


BEGIN_EVENT_TABLE(PickWindow, wxFrame)
  EVT_PAINT(PickWindow::OnPaint)
  EVT_SET_FOCUS(PickWindow::OnFocus)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(LogWindow, wxTextCtrl)
  EVT_KEY_DOWN(LogWindow::OnKeyboard)
  EVT_CHAR(LogWindow::OnKeyboard)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(GestureWindow, wxScrolledWindow)
  EVT_MOUSE_EVENTS(GestureWindow::OnMouse)
  EVT_SET_FOCUS(GestureWindow::OnFocus)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(SpellWindow, wxFrame)
  EVT_PAINT(SpellWindow::OnPaint)
  EVT_SET_FOCUS(SpellWindow::OnFocus)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(ArenaList, wxListBox)
  EVT_MOUSE_EVENTS(ArenaList::OnMouse)
  EVT_SET_FOCUS(ArenaList::OnFocus)
END_EVENT_TABLE()

